package fr.toutatice.cartoun.plugin.fragment;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.Documents;
import org.osivia.portal.api.cache.services.CacheInfo;
import org.osivia.portal.api.internationalization.Bundle;
import org.osivia.portal.api.internationalization.IBundleFactory;
import org.osivia.portal.api.internationalization.IInternationalizationService;
import org.osivia.portal.api.locator.Locator;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;

import fr.toutatice.cartoun.plugin.util.BapeListCommand;
import fr.toutatice.cartoun.plugin.util.CircoListCommand;
import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.PageSelectors;
import fr.toutatice.portail.cms.nuxeo.api.fragment.FragmentModule;
import fr.toutatice.portail.cms.nuxeo.api.services.NuxeoCommandContext;

/**
 * BAPE selector fragment module.
 *
 * @author Cédric Krommenhoek
 */
public class BapeSelectorFragmentModule extends FragmentModule {

    /** Selector label window property name. */
    private static final String LABEL_WINDOW_PROPERTY = "bape.selector.label";
    /** Selector tooltip label window property name. */
    private static final String TOOLTIP_LABEL_WINDOW_PROPERTY = "bape.selector.tooltip.label";
    /** Selector identifier window property name. */
    private static final String BAPE_SELECTOR_ID_WINDOW_PROPERTY = "bape.selector.id";

    /** Selector identifier window property name. */
    private static final String CIRCO_SELECTOR_ID_WINDOW_PROPERTY = "circo.selector.id";

    private static final String WEBID_PREFIX_BAPE = "bape_";

    private static final String WEBID_PREFIX_CIRCO = "circo_";

    /** JSP name. */
    private static final String JSP_NAME = "bape-selector";

    /** Internationalization bundle factory. */
    private IBundleFactory bundleFactory;


    /**
     * Constructor.
     *
     * @param portletContext portlet context
     */
    public BapeSelectorFragmentModule(PortletContext portletContext) {
        super(portletContext);

        // Internationalization bundle factory
        final IInternationalizationService internationalizationService = Locator.findMBean(IInternationalizationService.class,
                IInternationalizationService.MBEAN_NAME);
        this.bundleFactory = internationalizationService.getBundleFactory(this.getClass().getClassLoader());
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void doView(RenderRequest request, RenderResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Nuxeo controller
        final NuxeoController nuxeoController = new NuxeoController(request, response, portletContext);
        nuxeoController.setAuthType(NuxeoCommandContext.AUTH_TYPE_SUPERUSER);
        nuxeoController.setCacheType(CacheInfo.CACHE_SCOPE_PORTLET_CONTEXT);

        // Internationalization bundle
        final Bundle bundle = this.bundleFactory.getBundle(request.getLocale());

        // Domain path
        final String domainPath = nuxeoController.getDomainPath();

        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        // Label
        request.setAttribute("label", window.getProperty(LABEL_WINDOW_PROPERTY));
        // Tooltip label
        request.setAttribute("tooltipLabel", window.getProperty(TOOLTIP_LABEL_WINDOW_PROPERTY));

        if (StringUtils.isNotEmpty(domainPath)) {
            // Nuxeo command
            INuxeoCommand command = new BapeListCommand(domainPath);
            final Documents bapes = (Documents) nuxeoController.executeNuxeoCommand(command);

            command = new CircoListCommand(domainPath);
            final Documents circos = (Documents) nuxeoController.executeNuxeoCommand(command);


            if (bapes.isEmpty() && circos.isEmpty()) {
                request.setAttribute("osivia.emptyResponse", "1");
            }
            JSONArray data = new JSONArray();
            JSONObject selectEntry;

            // Selectors
            final String bapeSelectorId = window.getProperty(BAPE_SELECTOR_ID_WINDOW_PROPERTY);
            final String circoSelectorId = window.getProperty(CIRCO_SELECTOR_ID_WINDOW_PROPERTY);
            final Map<String, List<String>> selectors = PageSelectors.decodeProperties(request.getParameter("selectors"));
            String selectedId = null;
            // Selected BAPE identifier
            if (bapeSelectorId != null) {
                final List<String> values = selectors.get(bapeSelectorId);
                if (CollectionUtils.isNotEmpty(values)) {
                    selectedId = values.get(0);
                }
            }
            if (circoSelectorId != null) {
                final List<String> values = selectors.get(circoSelectorId);
                if (CollectionUtils.isNotEmpty(values)) {
                    selectedId = values.get(0);
                }
            }

            if (!bapes.isEmpty()) {
                JSONObject bapeGroup = new JSONObject();
                bapeGroup.put("id", "0");
                bapeGroup.put("text", bundle.getString("LABEL_BAPES"));
                JSONArray bapeChildren = new JSONArray();
                for (final Document bape : bapes) {
                    if (StringUtils.equals(selectedId, bape.getProperties().getString("ttc:webid"))) {
                        Map<String, String> bapeItem = new LinkedHashMap<String, String>();
                        bapeItem.put("id", bape.getProperties().getString("ttc:webid"));
                        bapeItem.put("text", bape.getTitle());
                        request.setAttribute("bapeItem", bapeItem);
                    } else {
                        selectEntry = new JSONObject();
                        selectEntry.put("id", bape.getProperties().getString("ttc:webid"));
                        selectEntry.put("text", bape.getTitle());
                        bapeChildren.add(selectEntry);
                    }
                }
                bapeGroup.put("children", bapeChildren);
                data.add(bapeGroup);
            }

            if (!circos.isEmpty()) {
                JSONObject circoGroup = new JSONObject();
                circoGroup.put("id", "1");
                circoGroup.put("text", bundle.getString("LABEL_CIRCO"));
                JSONArray circoChildren = new JSONArray();
                for (final Document circo : circos) {
                    if (StringUtils.equals(selectedId, circo.getProperties().getString("ttc:webid"))) {
                        Map<String, String> circoItem = new LinkedHashMap<String, String>();
                        circoItem.put("id", circo.getProperties().getString("ttc:webid"));
                        circoItem.put("text", circo.getTitle());
                        request.setAttribute("circoItem", circoItem);
                    } else {
                        selectEntry = new JSONObject();
                        selectEntry.put("id", circo.getProperties().getString("ttc:webid"));
                        selectEntry.put("text", circo.getTitle());
                        circoChildren.add(selectEntry);
                    }
                }
                circoGroup.put("children", circoChildren);
                data.add(circoGroup);
            }

            request.setAttribute("data", data.toString());
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void doAdmin(RenderRequest request, RenderResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Nuxeo controller
        new NuxeoController(request, response, portletContext);

        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        // Label
        request.setAttribute("label", window.getProperty(LABEL_WINDOW_PROPERTY));
        // Tooltip label
        request.setAttribute("tooltipLabel", window.getProperty(TOOLTIP_LABEL_WINDOW_PROPERTY));
        // bape Identifier
        request.setAttribute("selectorId", window.getProperty(BAPE_SELECTOR_ID_WINDOW_PROPERTY));
        // circo Identifier
        request.setAttribute("circoSelectorId", window.getProperty(CIRCO_SELECTOR_ID_WINDOW_PROPERTY));

    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void processAction(ActionRequest request, ActionResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        if (PortletMode.VIEW.equals(request.getPortletMode()) && "filter".equals(request.getParameter(ActionRequest.ACTION_NAME))) {
            // Identifier
            final String bapeSelectorId = window.getProperty(BAPE_SELECTOR_ID_WINDOW_PROPERTY);
            final String circoSelectorId = window.getProperty(CIRCO_SELECTOR_ID_WINDOW_PROPERTY);
            if ((bapeSelectorId != null) || (circoSelectorId != null)) {
                // Selected BAPE identifier
                final String selectedId = request.getParameter("selectedId");

                // Update selectors
                final Map<String, List<String>> selectors = PageSelectors.decodeProperties(request.getParameter("selectors"));
                if (request.getParameter("selectors") != null) {
                    response.setRenderParameter("lastSelectors", request.getParameter("selectors"));
                }

                selectors.remove(bapeSelectorId);
                selectors.remove(circoSelectorId);

                if (request.getParameter("clear") == null) {
                    if (StringUtils.startsWith(selectedId, WEBID_PREFIX_BAPE)) {
                        final List<String> values = new ArrayList<String>(1);
                        values.add(selectedId);
                        selectors.put(bapeSelectorId, values);
                    } else if (StringUtils.startsWith(selectedId, WEBID_PREFIX_CIRCO)) {
                        final List<String> values = new ArrayList<String>(1);
                        values.add(selectedId);
                        selectors.put(circoSelectorId, values);
                    }
                }
                response.setRenderParameter("selectors", PageSelectors.encodeProperties(selectors));
            }

        } else if ("admin".equals(request.getPortletMode().toString()) && "save".equals(request.getParameter(ActionRequest.ACTION_NAME))) {
            // Label
            window.setProperty(LABEL_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("label")));
            // Tooltip label
            window.setProperty(TOOLTIP_LABEL_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("tooltipLabel")));
            // bape Identifier
            window.setProperty(BAPE_SELECTOR_ID_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("bapeSelectorId")));
            // circo Identifier
            window.setProperty(CIRCO_SELECTOR_ID_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("circoSelectorId")));
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isDisplayedInAdmin() {
        return true;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getViewJSPName() {
        return JSP_NAME;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getAdminJSPName() {
        return JSP_NAME;
    }

}
