package fr.toutatice.cartoun.customizer.attributes;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.portlet.PortletException;
import javax.xml.XMLConstants;
import javax.xml.namespace.QName;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Element;
import org.jboss.portal.core.controller.ControllerCommand;
import org.jboss.portal.core.controller.ControllerContext;
import org.jboss.portal.core.controller.ControllerException;
import org.jboss.portal.core.model.portal.Page;
import org.jboss.portal.core.model.portal.command.render.RenderPageCommand;
import org.jboss.portal.core.model.portal.navstate.PageNavigationalState;
import org.jboss.portal.core.navstate.NavigationalStateContext;
import org.jboss.portal.core.theme.PageRendition;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.PropertyMap;
import org.osivia.portal.api.PortalException;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.html.DOM4JUtils;
import org.osivia.portal.api.locator.Locator;
import org.osivia.portal.api.theming.IAttributesBundle;
import org.osivia.portal.api.urls.IPortalUrlFactory;
import org.osivia.portal.core.cms.ICMSService;
import org.osivia.portal.core.cms.ICMSServiceLocator;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.cms.NuxeoDocumentContext;


public class SocialMetadataAttributesBundle implements IAttributesBundle {

    /** social metadata */
    private static final String SOCIAL_METADATA_NAME = "acrennes.sites.cartoun.socialMetadata.name";

    private static final String OSIVIA_CONTENT_PATH = "osivia.cms.contentPath";

    private static final String META_TAG = "meta";

    private static final String ATTRIBUTE_NAME = "name";

    private static final String ATTRIBUTE_PROPERTY = "property";

    private static final String ATTRIBUTE_CONTENT = "content";

    private static final String TWITTER_CARD_TYPE = "summary_large_image";

    private static final String TWITTER_SITE = System.getProperty("acrennes.sites.cartoun.twitter.site");

    private static final String FB_APP_ID = System.getProperty("acrennes.sites.cartoun.facebook.app.id");

    private static final String VIGNETTE_DEFAUT_WEBID = "vignette-par-defaut";

    /** urlFactory */
    private final IPortalUrlFactory urlFactory;

    /** CMS service locator. */
    private final ICMSServiceLocator cmsServiceLocator;

    /** attributes names. */
    private final Set<String> names;

    /**
     * Singleton Holder
     */
    private static class SocialMetadataAttributesBundleHolder {

        private static final SocialMetadataAttributesBundle INSTANCE = new SocialMetadataAttributesBundle();
    }

    /**
     * Singleton instance access.
     *
     * @return singleton instance
     */
    public static SocialMetadataAttributesBundle getInstance() {
        return SocialMetadataAttributesBundleHolder.INSTANCE;
    }

    /**
     * Constructor
     */
    private SocialMetadataAttributesBundle() {
        // URL factory
        urlFactory = Locator.findMBean(IPortalUrlFactory.class, IPortalUrlFactory.MBEAN_NAME);
        // CMS service locator
        cmsServiceLocator = Locator.findMBean(ICMSServiceLocator.class, "osivia:service=CmsServiceLocator");
        names = new TreeSet<String>();
        names.add(SOCIAL_METADATA_NAME);
    }

    @Override
    public void fill(RenderPageCommand renderPageCommand, PageRendition pageRendition, Map<String, Object> attributes) throws ControllerException {

        List<String> metadataList = new ArrayList<String>();

        // Current page
        Page page = renderPageCommand.getPage();
        // Controller context
        ControllerContext controllerContext = renderPageCommand.getControllerContext();
        // Content path
        String contentPath = computePath(controllerContext, page, OSIVIA_CONTENT_PATH);
        // Portal controller context
        PortalControllerContext portalControllerContext = new PortalControllerContext(controllerContext);

        try {
            if (contentPath != null) {
                final NuxeoDocumentContext documentContext = NuxeoController.getDocumentContext(portalControllerContext, contentPath);

                if ((documentContext.getType() != null)) {
                    if (StringUtils.equals(documentContext.getType().getName(), "UsageNum")) {
                        final Document activite = documentContext.getDoc();
                        String webId = activite.getProperties().getString("ttc:webid");
                        boolean shareable = BooleanUtils.isTrue(activite.getProperties().getBoolean("unum:shareable"));
                        if (StringUtils.isNotBlank(webId) && shareable) {
                            String webPath = NuxeoController.webIdToCmsPath(webId);
                            String canonicalURL = urlFactory.getPermaLink(portalControllerContext, null, null, webPath, IPortalUrlFactory.PERM_LINK_TYPE_SHARE);

                            String vignetteLink = buildUsageNUmVignetteUrl(portalControllerContext, activite);
                            String vignetteUrl = urlFactory.getBasePortalUrl(portalControllerContext) + vignetteLink;
                            generateMetaTwitter(metadataList, activite, vignetteUrl, portalControllerContext);
                            generateMetaOpenGraph(metadataList, activite, canonicalURL, vignetteUrl, portalControllerContext);
                        }
                    }
                }
            }
        } catch (PortalException e) {
            throw new ControllerException(e);
        } catch (PortletException e) {
            throw new ControllerException(e);
        }
        attributes.put(SOCIAL_METADATA_NAME, metadataList);
    }

    @Override
    public Set<String> getAttributeNames() {
        return names;
    }

    /**
     * retrieve the correct image to display and build an Url for it
     *
     * @param portalControllerContext
     * @param formation
     * @return
     * @throws PortletException
     */
    public String buildUsageNUmVignetteUrl(PortalControllerContext portalControllerContext, Document activite) throws PortletException {

        String urlVignette = StringUtils.EMPTY;
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext.getPortletCtx());

        final PropertyMap map = activite.getProperties().getMap("ttc:vignette");
        if (map != null) {
            urlVignette = nuxeoController.createFileLink(activite, "ttc:vignette");
        } else {
            urlVignette = getPictureLinkByWebId(nuxeoController, VIGNETTE_DEFAUT_WEBID);
        }

        return urlVignette;
    }


    public String buildStageVignetteUrl(PortalControllerContext portalControllerContext, Document portalPage) throws PortletException {
        String vignetteLink = StringUtils.EMPTY;

        NuxeoController nuxeoController = new NuxeoController(portalControllerContext.getPortletCtx());
        nuxeoController.setServletRequest(((ControllerContext) portalControllerContext.getControllerCtx()).getServerInvocation().getServerContext()
                .getClientRequest());
        PropertyMap pageVignette = portalPage.getProperties().getMap("ttcn:picture");
        if (pageVignette != null) {
            // si la portalPage possède une vignette, alors on utilise celle-la
            vignetteLink = nuxeoController.createFileLink(portalPage, "ttcn:picture");
        } else {
            // sinon, on tente de récupérer la vignette générique
            vignetteLink = getPictureLinkByWebId(nuxeoController, "vignette-greta");
        }

        return vignetteLink;
    }

    /**
     * generates a picture link based on webid
     *
     * @param nuxeoController
     * @param webId
     * @return
     * @throws PortletException
     */
    private String getPictureLinkByWebId(NuxeoController nuxeoController, String webId) throws PortletException {
        final String vignetteGeneriquePath = NuxeoController.webIdToFetchPath(webId);
        // CMS service
        ICMSService cmsService = cmsServiceLocator.getCMSService();
        String vignetteCmsPath;
        try {
            // on teste l'existence de l'image
            vignetteCmsPath = cmsService.adaptWebPathToCms(nuxeoController.getCMSCtx(), vignetteGeneriquePath);
        } catch (Exception e) {
            return StringUtils.EMPTY;
        }
        return nuxeoController.createPictureLink(vignetteCmsPath, "Original");
    }

    /**
     * generates meta html tags for twitter for the document
     *
     * @param metadata
     * @param documentMetaProperties
     * @throws ControllerException
     */
    private void generateMetaTwitter(List<String> metadataList, Document activite, String vignetteUrl,
            PortalControllerContext portalControllerContext) throws ControllerException {
        final PropertyMap properties = activite.getProperties();
        // twitter metadata
        // twitter:card
        metadataList.add(buildMetaTagWithName("twitter:card", TWITTER_CARD_TYPE));
        // twitter:site // @gretaDeBreetagne
        metadataList.add(buildMetaTagWithName("twitter:site", TWITTER_SITE));
        // twitter:title
        if (StringUtils.isNotBlank(properties.getString("dc:title"))) {
            metadataList.add(buildMetaTagWithName("twitter:title", properties.getString("dc:title")));
        }
        // twitter:description
        if (StringUtils.isNotBlank(properties.getString("unum:resume"))) {
            metadataList.add(buildMetaTagWithName("twitter:description", properties.getString("unum:resume")));
        }
        // twitter:image
        metadataList.add(buildMetaTagWithName("twitter:image", vignetteUrl));
    }


    /**
     * generates meta html tags for openGraph/facebook for the document
     *
     * @param portalControllerContext
     *
     * @param seoMetaProperties
     * @param documentMetaProperties
     * @param canonicalURL
     * @throws ControllerException
     */
    private void generateMetaOpenGraph(List<String> metadataList, Document stage, String canonicalUrl, String vignetteUrl,
            PortalControllerContext portalControllerContext) throws ControllerException {
        final PropertyMap properties = stage.getProperties();
        // openGraph / facebook metadata
        // fb:app_id
        metadataList.add(buildMetaTagWithProperty("fb:app_id", FB_APP_ID));
        // og:title
        if (StringUtils.isNotBlank(properties.getString("dc:title"))) {
            metadataList.add(buildMetaTagWithProperty("og:title", properties.getString("dc:title")));
        }
        // og:image
        metadataList.add(buildMetaTagWithProperty("og:image", vignetteUrl));
        // og:type
        metadataList.add(buildMetaTagWithProperty("og:type", "article"));
        // og:url
        metadataList.add(buildMetaTagWithProperty("og:url", canonicalUrl));
        // og:description
        if (StringUtils.isNotBlank(properties.getString("unum:resume"))) {
            metadataList.add(buildMetaTagWithProperty("og:description", properties.getString("unum:resume")));
        }
        // article:author
        if (StringUtils.isNotBlank(properties.getString("dc:creator"))) {
            metadataList.add(buildMetaTagWithProperty("article:author", properties.getString("dc:creator")));
        }
    }


    /**
     * Builds a <meta name="name" content="content"> tag
     *
     * @param name
     * @param content
     * @return the tag
     */
    private String buildMetaTagWithName(String name, String content) {

        final Element metaTag = DOM4JUtils.generateElement(META_TAG, null, null);
        DOM4JUtils.addAttribute(metaTag, ATTRIBUTE_NAME, name);
        DOM4JUtils.addAttribute(metaTag, ATTRIBUTE_CONTENT, content);

        return DOM4JUtils.write(metaTag);
    }

    /**
     * Builds a <meta property="property" content="content"> tag
     *
     * @param property
     * @param content
     * @return the tag
     */
    private String buildMetaTagWithProperty(String property, String content) {

        final Element metaTag = DOM4JUtils.generateElement(META_TAG, null, null);
        DOM4JUtils.addAttribute(metaTag, ATTRIBUTE_PROPERTY, property);
        DOM4JUtils.addAttribute(metaTag, ATTRIBUTE_CONTENT, content);

        return DOM4JUtils.write(metaTag);
    }

    /**
     * Utility method used to compute CMS and Content path.
     *
     * @param controllerContext
     * @param page
     * @param pathType
     * @return CMS or content path
     */
    private String computePath(ControllerContext controllerContext, Page page, String pathType) {
        // Navigational state context
        NavigationalStateContext navigationalStateContext = (NavigationalStateContext) controllerContext
                .getAttributeResolver(ControllerCommand.NAVIGATIONAL_STATE_SCOPE);
        // Page state
        PageNavigationalState pageState = navigationalStateContext.getPageNavigationalState(page.getId().toString());

        String[] sPath = null;
        if (pageState != null) {
            sPath = pageState.getParameter(new QName(XMLConstants.DEFAULT_NS_PREFIX, "osivia.cms.contentPath"));
        }
        if (ArrayUtils.isNotEmpty(sPath)) {
            return sPath[0];
        }

        return null;
    }
}
