package fr.toutatice.cartoun.portlet.detailactivite.util.pdf;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URLDecoder;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.osivia.portal.core.cms.BinaryDescription;
import org.osivia.portal.core.cms.BinaryDescription.Type;
import org.osivia.portal.core.cms.CMSBinaryContent;

import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.tool.xml.ElementList;
import com.itextpdf.tool.xml.Pipeline;
import com.itextpdf.tool.xml.XMLWorker;
import com.itextpdf.tool.xml.XMLWorkerHelper;
import com.itextpdf.tool.xml.html.TagProcessorFactory;
import com.itextpdf.tool.xml.html.Tags;
import com.itextpdf.tool.xml.parser.XMLParser;
import com.itextpdf.tool.xml.pipeline.css.CSSResolver;
import com.itextpdf.tool.xml.pipeline.css.CssResolverPipeline;
import com.itextpdf.tool.xml.pipeline.end.ElementHandlerPipeline;
import com.itextpdf.tool.xml.pipeline.html.HtmlPipeline;
import com.itextpdf.tool.xml.pipeline.html.HtmlPipelineContext;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.ResourceUtil;


/**
 * @author dorian
 */
public class PDFUtil {

    private static final Pattern cleanHTMLTags = Pattern.compile("<(OBJECT|object).+\\/(OBJECT|object>)>");

    private static final Pattern imgTagToXhtml = Pattern.compile("<(img|IMG)[^>]+(>)");

    private static final Pattern PATTERN_TYPE = Pattern.compile("type=([^&]+)&");

    private static final Pattern PATTERN_PATH = Pattern.compile("path=([^&]+)&");

    private static final Pattern PATTERN_INDEX = Pattern.compile("index=([^&]+)&");

    private static final Pattern PATTERN_CONTENT = Pattern.compile("content=([^&]+)&");

    private static final Pattern PATTERN_FIELDNAME = Pattern.compile("fieldName=([^&]+)&");

    private PDFUtil() {
    }

    /**
     * transforme une String contenant du HTML en ElementList PDF
     * n'accepte que du XHTML
     * utilise le NuxeoImageProvider afin de récuper les images de nuxeo
     * ignore les tags object (voir Tags.getHtmlTagProcessorFactory())
     *
     * @param stringToParse
     * @param nuxeoController
     * @return
     * @throws IOException
     */
    public static ElementList parseStringHTMLToElements(String stringToParse, NuxeoController nuxeoController) throws IOException {

        StringReader sr = new StringReader(stringToParse);
        ElementList elementList = new ElementList();

        HtmlPipelineContext htmlContext = new HtmlPipelineContext(new OsiviaCssapplier());

        TagProcessorFactory htmlTagProcessorFactory = Tags.getHtmlTagProcessorFactory();
        htmlContext.setTagFactory(htmlTagProcessorFactory);
        htmlContext.setAcceptUnknown(true);
        htmlContext.setImageProvider(new NuxeoImageProvider(nuxeoController));

        CSSResolver cssResolver = XMLWorkerHelper.getInstance().getDefaultCssResolver(true);

        Pipeline<?> pipeline = new CssResolverPipeline(cssResolver, new HtmlPipeline(htmlContext, new ElementHandlerPipeline(elementList, null)));

        XMLWorker worker = new XMLWorker(pipeline, true);

        XMLParser p = new XMLParser(worker);

        p.parse(sr);

        return elementList;
    }

    /**
     * retourne une cellule de tableau vide sans bordure
     *
     * @return
     */
    public static PdfPCell emptyCell() {
        PdfPCell cell = new PdfPCell();
        cell.setBorder(Rectangle.NO_BORDER);
        return cell;
    }

    /**
     * nettoie une String contenant du HTML afin de pouvoir utiliser le parser Itext
     *
     * @param htmlToSanitize
     * @return
     */
    public static String sanitizeHTML(String htmlToSanitize) {
        Matcher matcher = cleanHTMLTags.matcher(htmlToSanitize);
        String scenarioS = matcher.replaceAll(StringUtils.EMPTY);

        matcher = imgTagToXhtml.matcher(scenarioS);
        if (matcher.find()) {
            matcher.reset();
            StringBuilder sb = new StringBuilder();
            while (matcher.find()) {
                sb.append(scenarioS.substring(0, matcher.end() - 1));
                sb.append("></img>");
                sb.append(scenarioS.substring(matcher.end()));
            }
            scenarioS = sb.toString();
        }
        return scenarioS;
    }

    /**
     * fetch les données d'une image à partir d'une url binaryservlet
     *
     * @param src
     * @param nuxeoController
     * @return
     */
    public static byte[] getImage(String src, NuxeoController nuxeoController) {
        byte[] byteArray = null;
        InputStream is = null;
        try {
            Matcher matcher = PATTERN_TYPE.matcher(src);
            String type = null;
            if (matcher.find()) {
                type = matcher.group(1);
            }
            matcher = PATTERN_PATH.matcher(src);
            String path = null;
            if (matcher.find()) {
                path = matcher.group(1);
                path = URLDecoder.decode(path, "UTF-8");
            }
            CMSBinaryContent attachedPicture = null;
            if (Type.ATTACHED_PICTURE.equals(BinaryDescription.Type.valueOf(type))) {
                matcher = PATTERN_INDEX.matcher(src);
                String index = null;
                if (matcher.find()) {
                    index = matcher.group(1);
                }
                attachedPicture = nuxeoController.fetchAttachedPicture(path, index);
            } else if (Type.PICTURE.equals(BinaryDescription.Type.valueOf(type))) {
                matcher = PATTERN_CONTENT.matcher(src);
                String content = null;
                if (matcher.find()) {
                    content = matcher.group(1);
                }
                attachedPicture = nuxeoController.fetchPicture(path, content);
            } else if (Type.FILE.equals(BinaryDescription.Type.valueOf(type))) {
                matcher = PATTERN_FIELDNAME.matcher(src);
                String fieldName = null;
                if (matcher.find()) {
                    fieldName = matcher.group(1);
                }
                nuxeoController.setStreamingSupport(true);
                attachedPicture = nuxeoController.fetchFileContent(path, fieldName);
            } else if (Type.BLOB.equals(BinaryDescription.Type.valueOf(type))) {
                matcher = PATTERN_INDEX.matcher(src);
                String index = null;
                if (matcher.find()) {
                    index = matcher.group(1);
                }
                attachedPicture = ResourceUtil.getBlobHolderContent(nuxeoController, path, index);
            }
            if (attachedPicture.getStream() != null) {
                is = attachedPicture.getStream();
            } else {
                is = new FileInputStream(attachedPicture.getFile());
            }
            byteArray = IOUtils.toByteArray(is);

        } catch (IOException e) {
        } catch (Exception e) {
        } finally {
            IOUtils.closeQuietly(is);
        }
        return byteArray;
    }

}
