/**
 * 
 */
package org.osivia.migration.runners;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.automation.OperationContext;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentModelList;
import org.osivia.migration.transaction.LauncherTransactionHelper;

import fr.toutatice.ecm.platform.core.constants.ToutaticeNuxeoStudioConst;
import fr.toutatice.ecm.platform.core.helper.ToutaticeOperationHelper;
import fr.toutatice.ecm.platform.core.utils.exception.ToutaticeException;

/**
 * @author david
 *
 */
public class IdsRunner extends AbstractRunner {
	
	private static final Log log = LogFactory.getLog(IdsRunner.class);
	
	/** Total counter */
	private int totalDocs = 0;
	/** Treated docs. */
	private int treatedDocs = 0;
	
	
	public IdsRunner(CoreSession session){
		super(session);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void run() {
		log.debug("===== Beginning of Ids migration =====");
		long begin = System.currentTimeMillis();
		this.totalDocs = 0;
		this.treatedDocs = 0;

		// We iterate over Domains
		String domainsQuery = "select * from Domain";
		DocumentModelList domains = this.session.query(domainsQuery);
		this.totalDocs += domains.size();

		for (DocumentModel domain : domains) {
			if(domain.hasSchema(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE)){
				String id = (String) domain.getPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID);
				if(StringUtils.isBlank(id)){
					setIdOnLiveNVersions(domain);
				}
			}

			// We get working copies with empty id
			String livesWithEmptyIdQuery = String
					.format("select * from Document where ecm:ancestorId = '%s' and (ttc:webid ='' or ttc:webid is null) and %s",
							domain.getId(), EMPTY_ID_FILTER);

			DocumentModelList livesWithEmptyIdDocs = this.session
					.query(livesWithEmptyIdQuery);
			this.totalDocs += livesWithEmptyIdDocs.size();

			for (DocumentModel live : livesWithEmptyIdDocs) {
				// We set id for each working copy and its versions
				setIdOnLiveNVersions(live);
			}
		}
		// TO TEST: to flush cache with livesWithEmptyIdDocs
		this.session.save();

		long duration = System.currentTimeMillis() - begin;
		log.debug("===== End of Ids migration : "
				.concat(String.valueOf(this.treatedDocs)).concat(" on ")
				.concat(String.valueOf(this.totalDocs)).concat(": ").concat(String.valueOf(duration)).concat(" ms ====="));
	}
	
	/** 
	 * Set Id on live and its versions.
	 * 
	 * @param live
	 */
	private void setIdOnLiveNVersions(DocumentModel live) {
		if(live.hasSchema(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE)){
			LauncherTransactionHelper.checkNStartTransaction();
			
			try {
				live = setId(live);
				this.treatedDocs += 1;
				
				String id = (String) live.getPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID);
				log.debug("Id ".concat(id).concat(" set on live: ").concat(live.getPathAsString()));
				List<DocumentModel> versions = this.session.getVersions(live.getRef());
				
				this.totalDocs += versions.size();
				
				for(DocumentModel version : versions){
					version.putContextData(CoreSession.ALLOW_VERSION_WRITE, Boolean.TRUE);
					version.setPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID, id);
					this.session.saveDocument(version);
					
					this.treatedDocs += 1;
					log.debug(" and version ".concat(version.getPathAsString()));
				}
				// TO TEST: To invalidate caches and use less memory but:
				// caches with livesWithEmptyIdDocs will it be flused too?
				this.session.save();
				
				LauncherTransactionHelper.commitOrRollbackTransaction();
				
			} catch (Exception e) {
				LauncherTransactionHelper.setTransactionRollbackOnly();
				log.error(e);
			}
		}
	}
	
	/**
	 * @param doc
	 * @return id for doc.
	 */
	protected DocumentModel setId(DocumentModel doc){
		OperationContext context = new OperationContext(this.session);
		context.setInput(doc);
		Map<String, Object> param = new HashMap<String, Object>(1);
		param.put("chainSource", "notRecursive");
		try {
			return (DocumentModel) ToutaticeOperationHelper.callOperation(context, "Document.SetWebId", param);
		} catch (ToutaticeException e) {
			log.error("Id on document ".concat(doc.getPathAsString()).concat(" can not have be set: ")
					.concat(e.getMessage()));
			return null;
		}
	}

}
