/**
 * 
 */

package org.osivia.migration.service;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.security.auth.login.LoginContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.core.api.CoreInstance;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.security.SecurityConstants;
import org.nuxeo.runtime.api.Framework;
import org.nuxeo.runtime.model.ComponentContext;
import org.nuxeo.runtime.model.ComponentInstance;
import org.nuxeo.runtime.model.DefaultComponent;
import org.osivia.migration.runners.AbstractRunner;

/**
 * @author david
 */
public class RunnerLauncherService extends DefaultComponent {

    private static final Log log = LogFactory.getLog(RunnerLauncherService.class);

    /** Extension point. */
    private static final String RUNNERS_PT_EXT = "runners";
    /** Silent method. */
    private static final String SILENT_METHOD = "silentRun";
    /** Collection of runners. */
    private static final List<RunnerDescriptor> runnersDescriptors = new ArrayList<RunnerDescriptor>();

    /**
     * {@inheritDoc}
     */
    @Override
    public int getApplicationStartedOrder() {
        return 9999;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void registerContribution(Object contribution, String extensionPoint, ComponentInstance contributor) throws Exception {
        if (RUNNERS_PT_EXT.equals(extensionPoint)) {
            RunnerDescriptor runnerDescriptor = (RunnerDescriptor) contribution;
            runnersDescriptors.add(runnerDescriptor);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void applicationStarted(ComponentContext context) throws Exception {

        if (log.isDebugEnabled()) {
            log.debug("[Migrations]");
        }

        for (RunnerDescriptor runnerDescriptor : runnersDescriptors) {

            if (log.isDebugEnabled()) {
                log.debug("[" + runnerDescriptor.getRunnerId() + "] Started");
            }

            try {

                if (runnerDescriptor.isEnabled() && RunnerController.needToMigrate(runnerDescriptor)) {

                    LoginContext loginContext = Framework.login();
                    CoreSession session = CoreInstance.openCoreSession(null, SecurityConstants.SYSTEM_USERNAME);

                    try {

                        String runnerClass = runnerDescriptor.getClazz();
                        Class<?> runner = Class.forName(runnerClass);

                        Method silentRunMethod = getSilentRunMethod(runner.getMethods());

                        if (silentRunMethod != null) {

                            Class<?>[] constructorParams = {CoreSession.class};
                            Constructor<?> constructor = runner.getDeclaredConstructor(constructorParams);
                            Object runnerInstance = constructor.newInstance(session);

                            Object[] params = {Boolean.TRUE, AbstractRunner.FILTERED_SERVICES_LIST};
                            silentRunMethod.invoke(runnerInstance, params);

                            RunnerController.storeMigrationStatus(runnerDescriptor);

                        } else {
                            log.error("No 'silentRun' method for " + runner.getName());
                        }

                    } finally {
                        session.close();
                        loginContext.logout();
                    }

                }

                if (log.isDebugEnabled()) {
                    log.debug("[" + runnerDescriptor.getRunnerId() + "] Ended");
                }

            } catch (Exception e) {
                // Logs and go to next runner
                log.error("[" + runnerDescriptor.getRunnerId() + "] ERROR: ", e);

                if (log.isDebugEnabled()) {
                    log.debug("[" + runnerDescriptor.getRunnerId() + "] Aborted");
                }
            }
        }
    }

    /**
     * 
     * @param methods
     * @return the silentRun method.
     */
    private Method getSilentRunMethod(Method[] methods) {
        Method srMethod = null;

        List<Method> methodsList = Arrays.asList(methods);

        Iterator<Method> iterator = methodsList.iterator();
        boolean found = false;

        while (iterator.hasNext() && !found) {
            Method method = iterator.next();
            String nameMethod = method.getName();

            if (SILENT_METHOD.equals(nameMethod)) {
                Class<?>[] parameterTypes = method.getParameterTypes();
                if (parameterTypes != null && parameterTypes.length == 2) {
                    srMethod = method;
                    found = true;
                }
            }
        }

        return srMethod;
    }

}
