/**
 * 
 */
package org.osivia.migration.runners;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentException;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.PathRef;
import org.nuxeo.ecm.platform.uidgen.UIDGenerator;
import org.nuxeo.runtime.api.Framework;
import org.osivia.migration.service.RunnerLauncherService;
import org.osivia.migration.transaction.LauncherTransactionHelper;

import fr.toutatice.ecm.platform.automation.SetWebID.UnrestrictedSilentSetWebIdRunner;
import fr.toutatice.ecm.platform.core.constants.ToutaticeNuxeoStudioConst;
import fr.toutatice.ecm.platform.service.webid.TTCUIDGeneratorService;


/**
 * @author david
 *
 */
public class IdsRunner extends AbstractRunner {

    /** Logger. */
    private static final Log log = LogFactory.getLog(IdsRunner.class);

    /** Lives with no webIds query. */
    private static final String NO_DEFINED_WEBID_LIVES_QUERY = "select * from Document where (ttc:webid = '' or ttc:webid is NULL) and ecm:isProxy = 0 and ecm:isVersion = 0 ";

    /** Versions with no webIds query. */
    private static final String NO_DEFINED_WEBID_VERSIONS_QUERY = "select * from Document where (ttc:webid = '' or ttc:webid is NULL) and ecm:isVersion = 1 ";

    /** Documents having toutatice schema clause. */
    private static final String HAS_TTC_CLAUSE = " and ecm:primaryType not in ('DefaultSearch', 'PublicationRelation', 'ManagementRoot', 'AdministrativeStatusContainer', 'CommentRelation', 'Comment', 'uarDemImpressionFermee_cv', 'doc_acaren_faceted_search', 'MerEnsSMContentView_cv', 'SimpleTask', 'DefaultRelation', 'ConditionalTask', 'UserInvitationContainer', 'uarDemImpressionAGerer_cv', 'StepFolder', 'cv_unu_DUN_cv', 'uarDemImpressionEnAttenteValidation_cv', 'cv_clp_ContentOnline_cv', 'UserInvitation', 'Post', 'AdministrativeStatus', 'DocumentRouteModelsRoot', 'HiddenFolder', 'cv_le_Content_approved_cv', 'UserProfile', 'uarDemImpressionFermeeOpe_cv', 'uarDemImpressionTraitee_cv', 'pud_cv', 'Tag', 'DocumentRoute', 'Tagging', 'uarDemImpressionRecue_cv', 'cv_le_Content_all_cv', 'DocumentRouteStep', 'cv_clp_Content_cv', 'MonProfil', 'ConditionalStepFolder', 'cv_le_Content_submitted_cv', 'demande_cv', 'uarDemImpressionTerminee_cv', 'TaskRoot', 'uarDemImpressionEnCours_cv', 'DocumentRouteInstancesRoot', 'FollowLifeCycleTransitionTask', 'PublishTask', 'UserWorkspacesRoot', 'RouteNode', 'FacetedSearchDefault', 'TaskDoc', 'cv_le_Content_integrated_cv', 'MaintenanceConfiguration', 'uarDemImpressionEnRecharge_cv', 'CommentRoot') ";

    /** UserWorkspaces clauses. */
    private static final String NOT_UWS_CLAUSE = " and not(ecm:path startswith '/home' or ecm:path startswith '/userworkspace-domain') ";
    private static final String UWS_CLAUSE = " and ecm:path startswith '/home' ";

    /** Prent space clause. */
    private static final String PARENT_SPACE_CLAUSE = " and ecm:path startswith '%s' ";

    /** Includes Users workspaces (/home domain) parameter key. */
    private static final String INCLUDE_UWS_PARAM = "uws";
    /** Parameter key of space to treat. */
    private static final String SPACE_MARKER_PARAM = "_SPACE_";
    /** Only versions parameter key. */
    private static final String ONLY_VERSIONS_PARAM = "onlyVersions";


    /** For second batch pass: lives are yet migrated. */
    private boolean onlyVersions = false;

    /** User workspaces treatment indicator. */
    private boolean excludesUws = true;

    /** Space to treat. */
    private String spacePath = StringUtils.EMPTY;

    /** WebId generator. */
    private static UIDGenerator wIdGenerator;

    /** Getter for webId generator. */
    private static UIDGenerator getWebIdGenerator() {
        if (wIdGenerator == null) {
            TTCUIDGeneratorService service = (TTCUIDGeneratorService) Framework.getRuntime().getComponent(TTCUIDGeneratorService.ID);
            wIdGenerator = service.getDefaultUIDGenerator();
        }
        return wIdGenerator;
    }

    /** Documents in transaction counter. */
    private int nbDocsInTransac = 0;

    /** WebIds collisions. */
    private int nbCollisions = 0;

    /**
     * Constructor.
     * 
     * @param session
     */
    public IdsRunner(CoreSession session) {
        super(session);
    }

    @Override
    public void setParams(String reqParams) {
        String[] params = StringUtils.split(reqParams, ",");
        if (params != null) {
            for (int ind = 0; ind < params.length; ind++) {
                // User ws
                if (StringUtils.equals(params[ind], INCLUDE_UWS_PARAM)) {
                    this.excludesUws = false;
                }

                // Domain specified
                if (StringUtils.startsWith(params[ind], SPACE_MARKER_PARAM)) {
                    this.spacePath = StringUtils.substringAfter(params[ind], SPACE_MARKER_PARAM);
                }

                // Only versions
                if (StringUtils.equals(params[ind], ONLY_VERSIONS_PARAM)) {
                    this.onlyVersions = true;
                }
            }
        }
    }

    /**
     * Gets lives without webIds or only versions without webIds
     * (this last case is only for a second pass mode).
     */
    @Override
    public int setInputs(int limit) {
        // Results size
        int size = 0;

        // Just one Space to treat
        String spaceParentClause = StringUtils.isNotEmpty(this.spacePath) ? String.format(PARENT_SPACE_CLAUSE, this.spacePath) : StringUtils.EMPTY;
        // Take userws or not into account
        String uwsClause = this.excludesUws ? NOT_UWS_CLAUSE : UWS_CLAUSE;

        String extraClause = HAS_TTC_CLAUSE + spaceParentClause + uwsClause;

        // Include parent if there is space parent clause
        if (StringUtils.isNotEmpty(spaceParentClause)) {
            DocumentModel parentSpace = this.session.getDocument(new PathRef(this.spacePath));
            String psWId = (String) parentSpace.getPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID);
            if (StringUtils.isBlank(psWId)) {
                this.inputs.add(parentSpace);
            }
        }

        if (!this.onlyVersions) {
            // Time log
            final long bq = System.currentTimeMillis();

            // Get lives
            this.inputs = this.session.query(NO_DEFINED_WEBID_LIVES_QUERY + extraClause, limit);

            if (log.isDebugEnabled()) {
                final long eq = System.currentTimeMillis();
                log.debug("[Lives query]: " + String.valueOf(eq - bq) + " ms");
            }
        } else {
            // Time log
            final long bqv = System.currentTimeMillis();

            // Get versions
            this.inputs = this.session.query(NO_DEFINED_WEBID_VERSIONS_QUERY + extraClause, limit);

            if (log.isDebugEnabled()) {
                final long eqv = System.currentTimeMillis();
                log.debug("[Versions query]: " + String.valueOf(eqv - bqv) + " ms");
            }
        }
        size = this.inputs.size();

        this.totalInputs += size;
        return size;
    }

    /**
     * Migrates lives and its versions.
     */
    @Override
    public void run() throws ClientException {
        if (!this.onlyVersions) {
            // Migrate lives and its versions
            migrateLivesNVersions();
        } else {
            // Lives are yet migrated: migrate their versions
            migrateOnlyVersions();
        }

        // Collisions
        if (log.isDebugEnabled()) {
            log.debug("[----- Collisions number: " + String.valueOf(this.nbCollisions) + " -----]");
        }
    }


    /**
     * Migrate lives and their versions.
     */
    private void migrateLivesNVersions() {
        if (this.inputs.size() > 0) {

            if (log.isDebugEnabled()) {
                log.debug("[----- Migrating lives and their versions -----]");
            }

            // Logs
            final long begin = System.currentTimeMillis();

            LauncherTransactionHelper.checkNStartTransaction();

            try {
                for (DocumentModel live : this.inputs) {
                    // Generate webid
                    String webId = generateWebId(this.session, live);

                    // Set on live
                    setId(live, webId);

                    // Get live's versions to set live's webId
                    List<DocumentModel> versions = this.session.getVersions(live.getRef());
                    for (DocumentModel version : versions) {
                        setIdOnVersion(version, webId);
                    }

                    // To do not exceed ATOMIC_SIZE_TRANSACTION documents
                    if (this.nbDocsInTransac > RunnerLauncherService.ATOMIC_SIZE_TRANSACTION) {
                        this.nbDocsInTransac = 0;
                        break;
                    }
                }

                // Persist
                this.session.save();

            } catch (Exception e) {
                LauncherTransactionHelper.setTransactionRollbackOnly();
                log.error(e);
            } finally {
                LauncherTransactionHelper.commitOrRollbackTransaction();
            }

            if (log.isDebugEnabled()) {
                final long end = System.currentTimeMillis();
                log.debug("[----- Lives and their versions migrated: " + String.valueOf(end - begin) + " ms -----]");
            }
        }
    }

    /**
     * Migates only versions: their lives must have been migrated
     * (this is a second batch pass case).
     */
    private void migrateOnlyVersions() {
        if (this.inputs.size() > 0) {

            if (log.isDebugEnabled()) {
                log.debug("[----- Migrating only versions -----]");
            }

            // Logs
            final long begin = System.currentTimeMillis();

            LauncherTransactionHelper.checkNStartTransaction();

            try {
                for (DocumentModel version : this.inputs) {
                    // Get live webId
                    DocumentModel live = this.session.getWorkingCopy(version.getRef());
                    String webId = (String) live.getPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID);

                    if (StringUtils.isNotBlank(webId)) {
                        // Set live's webId on its version
                        setIdOnVersion(version, webId);
                    } else {
                        if (live != null) {
                            log.error("No webId on: " + live.getPathAsString());
                        }
                    }
                }

                // Persist
                this.session.save();

            } catch (Exception e) {
                LauncherTransactionHelper.setTransactionRollbackOnly();
                log.error(e);
            } finally {
                LauncherTransactionHelper.commitOrRollbackTransaction();
            }

            if (log.isDebugEnabled()) {
                final long end = System.currentTimeMillis();
                log.debug("[----- Versions migrated: " + String.valueOf(end - begin) + " ms -----]");
            }
        }
    }

    /**
     * Generates unique webId.
     * 
     * @param session
     * @param doc
     * @return unique webId.
     * @throws DocumentException
     */
    private String generateWebId(CoreSession session, DocumentModel doc) throws DocumentException {
        String webId = getWebIdGenerator().createUID(doc);
        while (UnrestrictedSilentSetWebIdRunner.isNotUnique(session, doc, webId)) {
            webId = getWebIdGenerator().createUID(doc);
            this.nbCollisions += 1;
        }
        return webId;
    }

    /**
     * Set webId on document.
     * 
     * @param doc
     * @param webid
     * @return
     */
    private void setId(DocumentModel doc, String webid) {
        if (doc != null) {
            // Logs
            final long begin = System.currentTimeMillis();
            if (log.isDebugEnabled()) {
                log.debug("Treating live: " + doc.getPathAsString());
            }

            doc.setPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID, webid);
            this.session.saveDocument(doc);

            // Count
            this.treatedInputs += 1;
            // Nb docs in transac
            this.nbDocsInTransac += 1;

            if (log.isDebugEnabled()) {
                final long end = System.currentTimeMillis();
                log.debug("SetId [" + webid + "] on live: " + String.valueOf(end - begin) + " ms");
            }
        }
    }

    /**
     * Set webId on version.
     * 
     * @param version
     * @param webid
     */
    private void setIdOnVersion(DocumentModel version, String webid) {
        if (version != null) {
            // Logs
            final long begin = System.currentTimeMillis();
            if (log.isDebugEnabled()) {
                log.debug("Treating version: " + version.getPathAsString());
            }

            // Set version as writable
            version.putContextData(CoreSession.ALLOW_VERSION_WRITE, true);

            version.setPropertyValue(ToutaticeNuxeoStudioConst.CST_DOC_SCHEMA_TOUTATICE_WEBID, webid);
            this.session.saveDocument(version);

            // Count
            this.treatedInputs += 1;
            this.totalInputs += 1;
            // Nb docs in transac
            this.nbDocsInTransac += 1;

            if (log.isDebugEnabled()) {
                final long end = System.currentTimeMillis();
                log.debug("SetId [" + webid + "] on version: " + String.valueOf(end - begin) + " ms");
            }
        }
    }


}
