/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.portal.identity.db;

import org.jboss.portal.identity.service.MembershipModuleService;
import org.jboss.portal.identity.User;
import org.jboss.portal.identity.Role;
import org.jboss.portal.identity.db.HibernateUserImpl;
import org.jboss.portal.identity.db.HibernateRoleImpl;
import org.jboss.portal.identity.IdentityException;
import org.jboss.portal.identity.CachedUserImpl;
import org.jboss.portal.identity.IdentityContext;
import org.jboss.portal.identity.NoSuchUserException;
import org.jboss.portal.common.util.Tools;
import org.hibernate.SessionFactory;
import org.hibernate.Session;
import org.hibernate.Query;
import org.hibernate.HibernateException;

import javax.naming.InitialContext;

import java.util.LinkedHashSet;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Collections;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at jboss.org">Boleslaw Dawidowicz</a>
 * @version $Revision: 1.1 $
 */
public class HibernateMembershipModuleImpl extends MembershipModuleService
{
   /** . */
   private static final org.jboss.logging.Logger log = org.jboss.logging.Logger.getLogger(HibernateMembershipModuleImpl.class);
   /** . */
   protected SessionFactory sessionFactory;

   /** . */
   protected String sessionFactoryJNDIName;

   private HibernateUserModuleImpl userModule;

   public void start() throws Exception
   {
      //
      sessionFactory = (SessionFactory)new InitialContext().lookup(sessionFactoryJNDIName);

      super.start();
   }

   public void stop() throws Exception
   {

      //
      sessionFactory = null;

      super.stop();
   }

   public String getSessionFactoryJNDIName()
   {
      return sessionFactoryJNDIName;
   }

   public void setSessionFactoryJNDIName(String sessionFactoryJNDIName)
   {
      this.sessionFactoryJNDIName = sessionFactoryJNDIName;
   }

   public Set getRoles(User user) throws IdentityException
   {
      if (user == null)
      {
         throw new IllegalArgumentException("User cannot be null");
      }

      if (user instanceof CachedUserImpl)
      {
         try
         {
            user = getUserModule().findUserById(user.getId());
         }
         catch(NoSuchUserException e)
         {
            throw new IdentityException("Illegal state - cached user doesn't exist in identity store: ", e);
         }
      }

      if (!(user instanceof HibernateUserImpl))
      {
         throw new IllegalArgumentException("User is not a HibernateUserImpl user");
      }

      // We return an immutable set to avoid modifications
      HibernateUserImpl ui = (HibernateUserImpl)user;
      Set roles = ui.getRoles();
      Set copy = new HashSet();
      for (Iterator iterator = roles.iterator(); iterator.hasNext();)
      {
         HibernateRoleImpl role = (HibernateRoleImpl)iterator.next();
         copy.add(role);
      }

      return Collections.unmodifiableSet(copy);
   }

   public Set getUsers(Role role) throws IdentityException
   {
      if (!(role instanceof HibernateRoleImpl))
      {
         throw new IllegalArgumentException("Role is not a HibernateRoleImpl role");
      }

      // We return an immutable set to avoid modifications
      HibernateRoleImpl ri = (HibernateRoleImpl)role;
      Set users = ri.getUsers();
      Set copy = new HashSet();
      for (Iterator iterator = users.iterator(); iterator.hasNext();)
      {
         HibernateUserImpl user = (HibernateUserImpl)iterator.next();
         copy.add(user);
      }

      return Collections.unmodifiableSet(copy);
   }

   public void assignUsers(Role role, Set users) throws IdentityException
   {
      //throw new UnsupportedOperationException("Not yet implemented");
      if (!(role instanceof HibernateRoleImpl))
      {
         throw new IllegalArgumentException("Role is not a HibernateRoleImpl role");
      }

      for (Iterator i = users.iterator(); i.hasNext();)
      {
         Object o = i.next();

         if (o instanceof CachedUserImpl)
         {
            try
            {
               o = getUserModule().findUserById(((User)o).getId());
            }
            catch(NoSuchUserException e)
            {
               log.error("Illegal state - cached user doesn't exist in identity store: ", e);
            }
         }

         if (o instanceof HibernateUserImpl)
         {
            HibernateUserImpl user = (HibernateUserImpl)o;
            user.getRoles().add(role);
         }
         else
         {
            throw new IllegalArgumentException("Only HibernateUserImpl users can be accepted");
         }
      }

      fireMembershipChangedEvent(role, users);

   }

   public void assignRoles(User user, Set roles) throws IdentityException
   {

      if (user == null)
      {
         throw new IllegalArgumentException("User cannot be null");
      }

      if (user instanceof CachedUserImpl)
      {
         try
         {
            user = getUserModule().findUserById(user.getId());
         }
         catch(NoSuchUserException e)
         {
            throw new IdentityException("Illegal state - cached user doesn't exist in identity store: ", e);
         }
      }

      if (!(user instanceof HibernateUserImpl))
      {
         throw new IllegalArgumentException("User is not a HibernateUserImpl user");
      }

      // We make a defensive copy with unwrapped maps and update with a new set
      Set copy = new HashSet();
      for (Iterator i = roles.iterator(); i.hasNext();)
      {
         Object o = i.next();
         if (o instanceof HibernateRoleImpl)
         {
            copy.add(o);
         }
         else
         {
            throw new IllegalArgumentException("Only HibernateRoleImpl roles can be accepted");
         }
      }

      // Assign new roles
      HibernateUserImpl ui = (HibernateUserImpl)user;
      ui.setRoles(copy);

      fireMembershipChangedEvent(user, roles);
   }

   //TODO:
   public Set findRoleMembers(String roleName, int offset, int limit, String userNameFilter) throws IdentityException
   {
      if (roleName != null)
      {
         try
         {
            Session session = getCurrentSession();

            Query query;
            if (userNameFilter.trim().length() != 0)
            {
               //
               userNameFilter = "%" + userNameFilter.replaceAll("%", "") + "%";

               //
               query = session.createQuery("from HibernateUserImpl as user left join user.roles role where role.name=:name" + " AND user.userName LIKE :filter order by user.userName");
               query.setString("filter", userNameFilter);
            }
            else
            {
               query = session.createQuery("from HibernateUserImpl as user left join user.roles role where role.name=:name order by user.userName");
            }
            query.setString("name", roleName);
            query.setFirstResult(offset);
            query.setMaxResults(limit);

            Iterator iterator = query.iterate();
            Set result = Tools.toSet(iterator, true);

            Set newResult = new LinkedHashSet();
            Iterator cleaner = result.iterator();
            while (cleaner.hasNext())
            {
               Object[] oArr = (Object[])cleaner.next();
               newResult.add(oArr[0]);
            }

            return newResult;
         }
         catch (HibernateException e)
         {
            String message = "Cannot find role  " + roleName;
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }
      else
      {
         throw new IllegalArgumentException("id cannot be null");
      }
   }

   protected HibernateUserModuleImpl getUserModule() throws IdentityException
   {

      if (userModule == null)
      {
         try
         {
            this.userModule = (HibernateUserModuleImpl)getIdentityContext().getObject(IdentityContext.TYPE_USER_MODULE);
         }
         catch (ClassCastException e)
         {
            throw new IdentityException("Not supported object as part of the context - must be HibernateUserModuleImpl", e);
         }
      }
      return userModule;
   }

   /** Can be subclasses to provide testing in a non JTA environment. */
   protected Session getCurrentSession()
   {
      if (sessionFactory == null)
      {
         throw new IllegalStateException("No session factory");
      }
      return sessionFactory.getCurrentSession();
   }
}
