/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.portal.identity.db;

import org.jboss.portal.identity.User;
import org.jboss.portal.identity.UserProfileModule;
import org.jboss.portal.identity.IdentityContext;
import org.jboss.portal.identity.info.ProfileInfo;
import org.jboss.portal.identity.info.PropertyInfo;
import org.jboss.portal.identity.db.HibernateUserImpl;
import org.hibernate.SessionFactory;
import org.hibernate.Session;
import org.hibernate.Query;
import org.hibernate.HibernateException;

import javax.naming.InitialContext;
import org.jboss.portal.identity.service.UserProfileModuleService;
import org.jboss.portal.identity.IdentityException;
import org.jboss.portal.identity.NoSuchUserException;
import org.jboss.portal.identity.UserModule;
import org.jboss.portal.identity.CachedUserImpl;

import java.util.Map;
import java.util.Set;
import java.util.Iterator;
import java.util.HashMap;
import java.util.Random;
import java.util.Collections;

/**
 * @author <a href="mailto:boleslaw.dawidowicz@jboss.org">Boleslaw Dawidowicz</a>
 * @version $Revision: 1.1 $
 */
public class HibernateUserProfileModuleImpl extends UserProfileModuleService
{

   /** . */
   private static final org.jboss.logging.Logger log = org.jboss.logging.Logger.getLogger(HibernateUserProfileModuleImpl.class);
   /** . */
   protected SessionFactory sessionFactory;

   protected UserModule userModule;

   /** . */
   protected String sessionFactoryJNDIName;

   private boolean synchronizeNonExistingUsers = true;

   private boolean enableSynchronizedUsers = true;

   private boolean acceptOtherImplementations = true;

   private String defaultSynchronizePassword;

   private boolean randomSynchronizePassword = false;

   public void start() throws Exception
   {
      //
      sessionFactory = (SessionFactory)new InitialContext().lookup(sessionFactoryJNDIName);

      super.start();

   }

   public void stop() throws Exception
   {
      //
      sessionFactory = null;
      super.stop();
   }

   public String getSessionFactoryJNDIName()
   {
      return sessionFactoryJNDIName;
   }

   public void setSessionFactoryJNDIName(String sessionFactoryJNDIName)
   {
      this.sessionFactoryJNDIName = sessionFactoryJNDIName;
   }

   public Object getProperty(User user, String propertyName) throws IdentityException
   {
      if (user == null)
      {
         throw new IllegalArgumentException("User cannot be null");
      }
      if (propertyName == null)
      {
         throw new IllegalArgumentException("Property name need to have value");
      }

      HibernateUserImpl dbUser = processUser(user);

      PropertyInfo pi = getProfileInfo().getPropertyInfo(propertyName);

      if (pi == null)
      {
         throw new IdentityException("Cannot find profile information about property: " + propertyName);
      }

      return dbUser.getProfileMap().get(propertyName);
   }

   public void setProperty(User user, String propertyName, Object propertyValue) throws IdentityException
   {
      if (user == null)
      {
         throw new IllegalArgumentException("User cannot be null");
      }
      if (propertyName == null)
      {
         throw new IllegalArgumentException("Property name need to have value");
      }

      HibernateUserImpl dbUser = processUser(user);


      PropertyInfo pi = getProfileInfo().getPropertyInfo(propertyName);

      if (pi == null)
      {
         throw new IdentityException("Cannot find profile information about property: " + propertyName);
      }
      else if (!pi.getAccessMode().equals(PropertyInfo.ACCESS_MODE_READ_WRITE))
      {
         throw new IdentityException("Property is not allowed for write access: " + propertyName);
      }

      if (propertyValue != null && !pi.getType().equals(propertyValue.getClass().getName()))
      {
         throw new IdentityException("Wrong property type. Must be: " + pi.getType() + "; and found: " + propertyValue.getClass().getName());
      }

      //if value is null reset property

      if (propertyValue != null)
      {
         dbUser.getProfileMap().put(propertyName, propertyValue);
      }
      else
      {
         dbUser.getProfileMap().remove(propertyName);
      }
      fireUserProfileChangedEvent(user.getId(), user.getUserName(), propertyName, propertyValue);
   }

   public Map getProperties(User user) throws IdentityException
   {
      if (user == null)
      {
         throw new IllegalArgumentException("User cannot be null");
      }

      HibernateUserImpl dbUser = processUser(user);



      //make a copy
      Map props = new HashMap();
      Map profile = dbUser.getProfileMap();
      Set keys = profile.keySet();
      for (Iterator iterator = keys.iterator(); iterator.hasNext();)
      {
         String key = (String)iterator.next();
         props.put(key, profile.get(key));
      }
      return Collections.unmodifiableMap(props);
   }



   /** Can be subclasses to provide testing in a non JTA environement. */
   protected Session getCurrentSession()
   {
      if (sessionFactory == null)
      {
         throw new IllegalStateException("No session factory");
      }
      return sessionFactory.getCurrentSession();
   }

   protected HibernateUserImpl processUser(User user) throws IdentityException
   {
      if (user instanceof CachedUserImpl)
      {
         try
         {
            user = getUserModule().findUserById(user.getId());
         }
         catch(NoSuchUserException e)
         {
            throw new IdentityException("Illegal state - cached user doesn't exist in identity store: ", e);
         }
      }

      if (user instanceof HibernateUserImpl)
      {
         return (HibernateUserImpl)user;
      }
      else if (!isAcceptOtherImplementations())
      {
         throw new IllegalArgumentException("This UserProfileModule implementation support only HibenrateUserImpl objects - set acceptOtherImplementations option to true");
      }
      if (log.isDebugEnabled()) log.debug("Processing non HibernateUserImpl object: " + user.getClass());
      //if not Hibernate user try to obtain it using userName
      Session session = getCurrentSession();
      Query query = session.createQuery("from HibernateUserImpl where userName=:userName");
      query.setParameter("userName", user.getUserName());
      query.setCacheable(true);
      HibernateUserImpl hu = (HibernateUserImpl)query.uniqueResult();

      if (hu != null )
      {
         return hu;
      }
      else if (!isSynchronizeNonExistingUsers())
      {
         throw new IdentityException("No user in DB - set synchronizeNonExistingUsers option to true");
      }
      else
      {
         try
         {
            hu = new HibernateUserImpl(user.getUserName());
            //user.updatePassword(user.getPassword());
            if (defaultSynchronizePassword != null)
            {
               hu.updatePassword(getDefaultSynchronizePassword());
            }
            //really dummy password generation
            //TODO: make something more sophisticated (risk of this part is documented)
            else if (randomSynchronizePassword)
            {
               Random r = new Random();
               StringBuffer password = new StringBuffer();
               for (int i = 0; i < 10; i++)
               {
                  password.append(r.nextDouble());
               }
               hu.updatePassword(password.toString());
            }
            session = getCurrentSession();

            //so if we synchronize from LDAP lets make the user enabled
            if (isEnableSynchronizedUsers())
            {
               hu.setEnabled(true);
            }
            session.save(hu);

            return hu;
         }
         catch (HibernateException e)
         {
            String message = "Cannot create user " + user.getUserName();
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }


   }


   /**
    * obtains UserProfile object - if module is used as a Delegate it tries to obtain it from the main one.
    * @return
    * @throws IdentityException
    */
   public ProfileInfo getProfileInfo() throws IdentityException
   {


      if (profileInfo == null)
      {
         //obtain main UserProfileModule
         UserProfileModule module = (UserProfileModule)getIdentityContext().getObject(IdentityContext.TYPE_USER_PROFILE_MODULE);
         if (module == this)
         {
            throw new IdentityException("ProfileInfo not accessible - check configuration");
         }
         else
         {
            setProfileInfo(module.getProfileInfo());
         }
      }
      return profileInfo;
   }


   public boolean isSynchronizeNonExistingUsers()
   {
      return synchronizeNonExistingUsers;
   }

   public void setSynchronizeNonExistingUsers(boolean synchronizeNonExistingUsers)
   {
      this.synchronizeNonExistingUsers = synchronizeNonExistingUsers;
   }

   public boolean isEnableSynchronizedUsers()
   {
      return enableSynchronizedUsers;
   }

   public void setEnableSynchronizedUsers(boolean enableSynchronizedUsers)
   {
      this.enableSynchronizedUsers = enableSynchronizedUsers;
   }

   public boolean isAcceptOtherImplementations()
   {
      return acceptOtherImplementations;
   }

   public void setAcceptOtherImplementations(boolean acceptOtherImplementations)
   {
      this.acceptOtherImplementations = acceptOtherImplementations;
   }

   public String getDefaultSynchronizePassword()
   {
      return defaultSynchronizePassword;
   }

   public void setDefaultSynchronizePassword(String defaultSynchronizePassword)
   {
      this.defaultSynchronizePassword = defaultSynchronizePassword;
   }


   public boolean isRandomSynchronizePassword()
   {
      return randomSynchronizePassword;
   }

   public void setRandomSynchronizePassword(boolean randomSynchronizePassword)
   {
      this.randomSynchronizePassword = randomSynchronizePassword;
   }

   protected UserModule getUserModule() throws IdentityException
   {
      if (userModule == null)
      {
         try
         {
            this.userModule = (UserModule)getIdentityContext().getObject(IdentityContext.TYPE_USER_MODULE);
         }
         catch (ClassCastException e)
         {
            throw new IdentityException("Not supported object as part of the context - must be UserModule", e);
         }
      }
      return userModule;
   }
}

