package fr.toutatice.cartoun.plugin.fragment;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.Documents;
import org.osivia.portal.api.cache.services.CacheInfo;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;

import fr.toutatice.cartoun.plugin.util.BapeListCommand;
import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.PageSelectors;
import fr.toutatice.portail.cms.nuxeo.api.domain.DocumentDTO;
import fr.toutatice.portail.cms.nuxeo.api.fragment.FragmentModule;
import fr.toutatice.portail.cms.nuxeo.api.services.NuxeoCommandContext;
import fr.toutatice.portail.cms.nuxeo.api.services.dao.DocumentDAO;

/**
 * BAPE selector fragment module.
 *
 * @author Cédric Krommenhoek
 */
public class BapeSelectorFragmentModule extends FragmentModule {

    /** Selector label window property name. */
    private static final String LABEL_WINDOW_PROPERTY = "bape.selector.label";
    /** Selector identifier window property name. */
    private static final String SELECTOR_ID_WINDOW_PROPERTY = "bape.selector.id";

    /** JSP name. */
    private static final String JSP_NAME = "bape-selector";


    /** Document DAO. */
    private final DocumentDAO documentDao;


    /**
     * Constructor.
     *
     * @param portletContext portlet context
     */
    public BapeSelectorFragmentModule(PortletContext portletContext) {
        super(portletContext);

        // Document DAO
        this.documentDao = DocumentDAO.getInstance();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void doView(RenderRequest request, RenderResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Nuxeo controller
        final NuxeoController nuxeoController = new NuxeoController(request, response, portletContext);
        nuxeoController.setAuthType(NuxeoCommandContext.AUTH_TYPE_SUPERUSER);
        nuxeoController.setCacheType(CacheInfo.CACHE_SCOPE_PORTLET_CONTEXT);

        // Domain path
        final String domainPath = nuxeoController.getDomainPath();

        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        // Label
        request.setAttribute("label", window.getProperty(LABEL_WINDOW_PROPERTY));

        if (StringUtils.isNotEmpty(domainPath)) {
            // Nuxeo command
            final INuxeoCommand command = new BapeListCommand(domainPath);
            final Documents documents = (Documents) nuxeoController.executeNuxeoCommand(command);

            final List<DocumentDTO> pools = new ArrayList<DocumentDTO>(documents.size());
            for (final Document document : documents) {
                final DocumentDTO pool = this.documentDao.toDTO(document);
                pools.add(pool);
            }
            request.setAttribute("pools", pools);

            // Selectors
            final String selectorId = window.getProperty(SELECTOR_ID_WINDOW_PROPERTY);
            if (selectorId != null) {
                final Map<String, List<String>> selectors = PageSelectors.decodeProperties(request.getParameter("selectors"));
                final List<String> values = selectors.get(selectorId);

                // Selected BAPE identifier
                if (CollectionUtils.isNotEmpty(values)) {
                    final String selectedId = values.get(0);
                    if (selectedId != null) {
                        for (final DocumentDTO pool : pools) {
                            if (selectedId.equals(pool.getProperties().get("ttc:webid"))) {
                                pool.getProperties().put("selected", true);
                                break;
                            }
                        }
                    }
                }
            }
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void doAdmin(RenderRequest request, RenderResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Nuxeo controller
        new NuxeoController(request, response, portletContext);

        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        // Label
        request.setAttribute("label", window.getProperty(LABEL_WINDOW_PROPERTY));
        // Identifier
        request.setAttribute("selectorId", window.getProperty(SELECTOR_ID_WINDOW_PROPERTY));
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected void processAction(ActionRequest request, ActionResponse response, PortletContext portletContext) throws PortletException, IOException {
        // Current window
        final PortalWindow window = WindowFactory.getWindow(request);

        if (PortletMode.VIEW.equals(request.getPortletMode()) && "filter".equals(request.getParameter(ActionRequest.ACTION_NAME))) {
            // Identifier
            final String selectorId = window.getProperty(SELECTOR_ID_WINDOW_PROPERTY);
            if (selectorId != null) {
                // Selected BAPE identifier
                final String selectedId = request.getParameter("selectedId");

                // Update selectors
                final Map<String, List<String>> selectors = PageSelectors.decodeProperties(request.getParameter("selectors"));
                if (StringUtils.isEmpty(selectedId)) {
                    selectors.remove(selectorId);
                } else {
                    final List<String> values = new ArrayList<String>(1);
                    values.add(request.getParameter("selectedId"));
                    selectors.put(selectorId, values);
                }
                response.setRenderParameter("selectors", PageSelectors.encodeProperties(selectors));
            }

        } else if ("admin".equals(request.getPortletMode().toString()) && "save".equals(request.getParameter(ActionRequest.ACTION_NAME))) {
            // Label
            window.setProperty(LABEL_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("label")));
            // Identifier
            window.setProperty(SELECTOR_ID_WINDOW_PROPERTY, StringUtils.trimToNull(request.getParameter("selectorId")));
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isDisplayedInAdmin() {
        return true;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getViewJSPName() {
        return JSP_NAME;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getAdminJSPName() {
        return JSP_NAME;
    }

}
