package fr.toutatice.cartoun.generator.repository;

import io.codearte.jfairy.Fairy;
import io.codearte.jfairy.producer.company.Company;
import io.codearte.jfairy.producer.person.Person;
import io.codearte.jfairy.producer.text.TextProducer;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.Set;

import javax.naming.NameAlreadyBoundException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapName;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.automation.client.OperationRequest;
import org.nuxeo.ecm.automation.client.Session;
import org.nuxeo.ecm.automation.client.adapters.DocumentService;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.PathRef;
import org.nuxeo.ecm.automation.client.model.PropertyMap;

import fr.toutatice.cartoun.generator.model.Academy;
import fr.toutatice.cartoun.generator.model.Configuration;
import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;

/**
 * Generate command.
 *
 * @author Cédric Krommenhoek
 * @see INuxeoCommand
 */
public class GenerateCommand implements INuxeoCommand {

    /** String list separator. */
    private static final String SEPARATOR = ",";

    /** Log. */
    private final Log log;
    /** LDAP context. */
    private final InitialLdapContext ldapContext;
    /** Generator configuration. */
    private final Configuration configuration;
    /** Academy. */
    private final Academy academy;
    /** Locale. */
    private final Locale locale;

    /** Errors count. */
    private int errors;


    /**
     * Constructor.
     *
     * @param ldapContext LDAP context
     * @param configuration generator configuration
     * @param academy academy
     * @param locale locale
     */
    public GenerateCommand(InitialLdapContext ldapContext, Configuration configuration, Academy academy, Locale locale) {
        super();
        this.log = LogFactory.getLog(this.getClass());
        this.ldapContext = ldapContext;
        this.configuration = configuration;
        this.academy = academy;
        this.locale = locale;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Object execute(Session nuxeoSession) throws Exception {
        long start = System.currentTimeMillis();
        this.log.info("[" + this.academy.getLabel() + "] start");

        DocumentService documentService = nuxeoSession.getAdapter(DocumentService.class);

        String path = "/" + this.academy.getName();
        boolean exists = true;
        try {
            documentService.getDocument(new PathRef(path));
        } catch (Exception e) {
            exists = false;
        }

        if (!exists) {
            // Academy
            Document academy = this.createAcademy(nuxeoSession, documentService);
            // Pools folder
            Document poolsFolder = this.createStructuresFolder(nuxeoSession, documentService, academy, "Bapes");
            // Pools
            List<Document> pools = this.createPools(nuxeoSession, documentService, poolsFolder);
            // Institutions folder
            Document institutionsFolder = this.createStructuresFolder(nuxeoSession, documentService, academy, "Etablissements");
            // Institutions
            List<Document> institutions = this.createInstitutions(nuxeoSession, documentService, institutionsFolder, pools);
            // Profiles
            List<String> profiles = this.createProfiles();
            // Users
            List<String> users = this.createUsers(institutions, profiles);
            // Service
            Document service = this.createService(nuxeoSession, documentService, academy);
            // Folders
            List<Document> folders = this.createFolders(nuxeoSession, documentService, service);
            // Activities
            this.createActivities(nuxeoSession, documentService, folders, users);
        }

        this.log.info("[" + this.academy.getLabel() + "] done (" + (System.currentTimeMillis() - start) + "ms)");

        if (this.errors > 0) {
            this.log.warn(String.valueOf(this.errors) + " erreurs");
        }

        return null;
    }


    /**
     * Create academies.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @return academies
     * @throws Exception
     */
    private Document createAcademy(Session nuxeoSession, DocumentService documentService) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création de l'académie");

        Document root = documentService.getRootDocument();

        PropertyMap properties = new PropertyMap();
        properties.set("dc:title", "ACA-" + this.academy.getCode());
        properties.set("dc:description", this.academy.getLabel());
        properties.set("ttc:webid", "aca_" + this.academy.getCode());

        return this.createDocument(nuxeoSession, documentService, root, "Domain", this.academy.getName(), properties);
    }


    /**
     * Create structures folder.
     * 
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param academy parent academy
     * @param name structures folder name
     * @return structures folder
     * @throws Exception
     */
    private Document createStructuresFolder(Session nuxeoSession, DocumentService documentService, Document academy, String name) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création de la structure '" + name + "'");

        PropertyMap properties = new PropertyMap();
        properties.set("dc:title", name);

        return this.createDocument(nuxeoSession, documentService, academy, "Structures", StringUtils.lowerCase(name), properties);
    }


    /**
     * Create pools.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param poolsFolder parent pools folder
     * @return pools
     * @throws Exception
     */
    private List<Document> createPools(Session nuxeoSession, DocumentService documentService, Document poolsFolder) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des bassins");

        Fairy fairy = Fairy.create(this.locale);
        TextProducer textProducer = fairy.textProducer();
        Random random = new Random();

        List<Document> pools = new ArrayList<Document>(this.configuration.getPoolsCount());

        for (int i = 0; i < this.configuration.getPoolsCount(); i++) {
            Person person = fairy.person();

            // Name
            String name = "bape-" + String.valueOf(i + 1);
            // Display name
            String displayName = "Bassin " + StringUtils.leftPad(String.valueOf(i + 1), 2, "0");
            // RNE
            String rne = StringUtils.leftPad(String.valueOf(random.nextInt(9999999)), 7, "0") + String.valueOf((char) (random.nextInt(26) + 'A'));

            // Properties
            PropertyMap properties = new PropertyMap();
            properties.set("dc:title", displayName);
            properties.set("ttc:webid", "bape_" + rne);
            properties.set("bape:ENTDisplayName", displayName);
            properties.set("bape:description", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
            properties.set("bape:ENTStructureTypeStruct", textProducer.word(1).replaceAll(",|;", StringUtils.EMPTY));
            properties.set("bape:ENTCodeAcademie", "ACA-" + this.academy.getCode());
            properties.set("bape:ou", rne);
            properties.set("bape:l", person.getAddress().getCity());

            try {
                // Permissions
                List<Permission> permissions = new ArrayList<Permission>(1);
                permissions.add(new Permission("personnel-tous", "Read"));
                
                Document pool = this.createDocument(nuxeoSession, documentService, poolsFolder, "ENTbape", name, properties, permissions);

                pools.add(pool);
            } catch (Exception e) {
                // Do nothing
            }
        }

        return pools;
    }


    /**
     * Create institutions.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param institutionsFolder parent institutions folder
     * @param pools parent pools
     * @return institutions
     * @throws Exception
     */
    private List<Document> createInstitutions(Session nuxeoSession, DocumentService documentService, Document institutionsFolder, List<Document> pools)
            throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des établissements");

        Fairy fairy = Fairy.create(this.locale);
        TextProducer textProducer = fairy.textProducer();
        Random random = new Random();

        int count = this.configuration.getInstitutionsCount() / this.configuration.getPoolsCount();
        int progression = 1;

        List<Document> institutions = new ArrayList<Document>(this.configuration.getInstitutionsCount());

        for (Document pool : pools) {
            for (int i = 0; i < count; i++) {
                if ((progression % 10) == 0) {
                    this.log.info("[" + this.academy.getLabel() + "] Création des établissements (" + String.valueOf(progression) + "/"
                            + String.valueOf(this.configuration.getInstitutionsCount()) + ")");
                }
                progression++;

                Person person = fairy.person();

                // Name
                String name = "etb-" + String.valueOf(i + 1) + "-" + StringUtils.substringAfterLast(pool.getPath(), "/");
                // Display name
                String displayName = "Établissement " + StringUtils.leftPad(String.valueOf(i + 1), 2, "0") + " (" + pool.getTitle() + ")";
                // RNE
                String rne = StringUtils.leftPad(String.valueOf(random.nextInt(9999999)), 7, "0") + String.valueOf((char) (random.nextInt(26) + 'A'));

                // Properties
                PropertyMap properties = new PropertyMap();
                properties.set("dc:title", displayName);
                properties.set("ttc:webid", "etb_" + rne);
                properties.set("etb:ENTDisplayName", displayName);
                properties.set("etb:ENTStructureNomCourant", textProducer.word(1).replaceAll(",|;", StringUtils.EMPTY));
                properties.set("etb:description", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("etb:ENTEtablissementBassin", pool.getString("ttc:webid"));
                properties.set("etb:ENTEtablissementContrat", person.passportNumber());
                properties.set("etb:ENTEtablissementMinistereTutelle", textProducer.word(1).replaceAll(",|;", StringUtils.EMPTY));
                properties.set("etb:ou", rne);
                properties.set("etb:ENTCodeAcademie", "ACA-" + this.academy.getCode());
                properties.set("etb:ENTOrganisationGeoLoc", this.generateGeolocation(random));
                properties.set("etb:ENTStructureTypeStruct", textProducer.word(1).replaceAll(",|;", StringUtils.EMPTY));
                properties.set("etb:telephoneNumber", person.telephoneNumber());
                properties.set("etb:facsimileTelephoneNumber", person.telephoneNumber());
                properties.set("etb:street", person.getAddress().street());
                properties.set("etb:postalCode", person.getAddress().getPostalCode());
                properties.set("etb:l", person.getAddress().getCity());

                try {
                    Document institution = this.createDocument(nuxeoSession, documentService, institutionsFolder, "ENTEtablissement", name, properties);
                    institutions.add(institution);
                } catch (Exception e) {
                    // Do nothing
                }
            }
        }

        return institutions;
    }


    /**
     * Create profiles.
     * 
     * @return profiles
     * @throws Exception
     */
    private List<String> createProfiles() throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des profils");

        List<String> profiles = new ArrayList<String>(this.configuration.getDisciplines().size() * this.configuration.getProfiles().size());

        for (String discipline : this.configuration.getDisciplines()) {
            for (String profile : this.configuration.getProfiles()) {
                // CN
                String cn = this.academy.getCode() + "-" + discipline + "-" + profile;
                // Display name
                String displayName = discipline + " - " + profile + " (" + academy.getLabel() + ")";
                
                // LDAP name
                LdapName ldapName = new LdapName(this.configuration.getProfilesBaseDn());
                ldapName.add("cn=" + cn);

                // LDAP attributes
                Attributes attributes = new BasicAttributes();
                attributes.put("ENTProfilPeuplement", "EXPLICITE");
                attributes.put("ENTProfilType", "space-group");
                attributes.put("ENTDisplayName", displayName);

                // Object class
                Attribute objectClass = new BasicAttribute("objectclass");
                objectClass.add("groupOfNames");
                objectClass.add("top");
                objectClass.add("ENTProfil");
                attributes.put(objectClass);

                try {
                    this.ldapContext.createSubcontext(ldapName, attributes);
                    profiles.add(ldapName.toString());
                } catch (NameAlreadyBoundException e) {
                    profiles.add(ldapName.toString());
                } catch (Exception e) {
                    this.log.error(e.getMessage(), e);
                }
            }
        }

        return profiles;
    }


    /**
     * Create users.
     *
     * @param institutions institutions
     * @param profiles profiles
     * @return users
     * @throws Exception
     */
    private List<String> createUsers(List<Document> institutions, List<String> profiles) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des utilisateurs");

        Fairy fairy = Fairy.create(this.locale);
        TextProducer textProducer = fairy.textProducer();
        Random random = new Random();

        List<String> users = new ArrayList<String>(this.configuration.getUsersCount());

        for (int i = 0; i < this.configuration.getUsersCount(); i++) {
            Person person = fairy.person();

            // UID
            String username = person.username().replaceAll(" ", StringUtils.EMPTY);
            String uid = username + "@" + this.academy.getCode();

            // User institutions
            float p = random.nextFloat();
            int count;
            if (p > 0.9) {
                count = 3;
            } else if (p > 0.75) {
                count = 2;
            } else {
                count = 1;
            }
            Set<String> userInstitutions = new HashSet<String>(count);
            for (int j = 0; j < count; j++) {
                Document institution = institutions.get(random.nextInt(institutions.size()));
                userInstitutions.add(StringUtils.removeStart(institution.getString("ttc:webid"), "etb_"));
            }

            // User profiles
            String profile;
            p = random.nextFloat();
            if (p > 0.1) {
                profile = profiles.get(random.nextInt(profiles.size()));
            } else {
                profile = null;
            }

            // LDAP name
            LdapName ldapName = new LdapName(this.configuration.getUsersBaseDn());
            ldapName.add("uid=" + uid);

            // LDAP attributes
            Attributes attributes = new BasicAttributes();
            attributes.put("cn", person.fullName());
            attributes.put("sn", person.lastName());
            attributes.put("givenName", person.firstName());
            attributes.put("displayName", person.fullName());
            attributes.put("userPassword", "osivia");
            attributes.put("mail", "lbillon+" + username + "@osivia.com");
            attributes.put("title", person.isFemale() ? "Mme" : "M.");
            attributes.put("ENTCodeAcademie", this.academy.getCode());
            attributes.put("ENTPersonAlias", textProducer.word(1));
            attributes.put("ENTPersonDateNaissance", person.dateOfBirth().toString());
            attributes.put("ENTPersonListeRouge", String.valueOf(random.nextBoolean()));
            if (profile != null) {
                // Add profile to user
                attributes.put("ENTPersonProfils", profile);

                // Update profile
                Attribute members = new BasicAttribute("member");
                members.add(ldapName.toString());
                Attribute explicitMembers = new BasicAttribute("ExplicitMember");
                explicitMembers.add(ldapName.toString());
                ModificationItem[] modifications = new ModificationItem[2];
                modifications[0] = new ModificationItem(DirContext.ADD_ATTRIBUTE, members);
                modifications[1] = new ModificationItem(DirContext.ADD_ATTRIBUTE, explicitMembers);
                this.ldapContext.modifyAttributes(profile, modifications);
            }

            // Object class
            Attribute objectClass = new BasicAttribute("objectclass");
            objectClass.add("person");
            objectClass.add("inetOrgPerson");
            objectClass.add("organizationalPerson");
            objectClass.add("top");
            objectClass.add("ENTPerson");
            attributes.put(objectClass);

            // RNE
            Attribute rne = new BasicAttribute("Rne");
            for (String institution : userInstitutions) {
                rne.add(institution);
            }
            attributes.put(rne);

            // Fonctions
            Attribute fonctions = new BasicAttribute("ENTPersonFonctions");
            List<String> list = Arrays.asList(StringUtils.split(textProducer.word(random.nextInt(3) + 1)));
            for (String fonction : list) {
                fonctions.add(fonction);
            }
            attributes.put(fonctions);

            try {
                this.ldapContext.createSubcontext(ldapName, attributes);
                users.add(ldapName.toString());
            } catch (NameAlreadyBoundException e) {
                users.add(ldapName.toString());
            } catch (Exception e) {
                this.log.error(e.getMessage(), e);
            }
        }

        return users;
    }


    /**
     * Create service.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param academy parent academy
     * @return service
     * @throws Exception
     */
    private Document createService(Session nuxeoSession, DocumentService documentService, Document academy) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création du SAP");

        // Name
        String name = "sap";

        // Properties
        PropertyMap properties = new PropertyMap();
        properties.set("dc:title", this.academy.getLabel());
        properties.set("ttc:webid", "sun_" + this.academy.getCode());
        properties.set("ttc:isPreloadedOnLogin", true);
        properties.set("ttc:tabOrder", String.valueOf(NumberUtils.toInt(this.academy.getCode())));

        // Permissions
        List<Permission> permissions = new ArrayList<Permission>(1);
        permissions.add(new Permission("personnel-tous", "Read"));

        return this.createDocument(nuxeoSession, documentService, academy, "UsageNumServiceMetier", name, properties, true, permissions);
    }


    /**
     * Create folders.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param service parent service
     * @return folders
     * @throws Exception
     */
    private List<Document> createFolders(Session nuxeoSession, DocumentService documentService, Document service) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des DAP");

        Fairy fairy = Fairy.create(this.locale);

        List<Document> folders = new ArrayList<Document>(this.configuration.getFoldersCount());

        for (int i = 0; i < this.configuration.getFoldersCount(); i++) {
            TextProducer textProducer = fairy.textProducer();

            // Name
            String name = "dap-" + String.valueOf(i + 1);

            // Properties
            PropertyMap properties = new PropertyMap();
            properties.set("dc:title", "DAP " + StringUtils.leftPad(String.valueOf(i + 1), 2, "0"));
            properties.set("ttc:webid", "dun_" + this.academy.getCode() + "_" + name);
            properties.set("dun:subtitle", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));

            try {
                // Permissions
                List<Permission> permissions = new ArrayList<GenerateCommand.Permission>(this.configuration.getDisciplines().size()
                        * this.configuration.getProfiles().size());
                for (String discipline : this.configuration.getDisciplines()) {
                    for (String profile : this.configuration.getProfiles()) {
                        String user = this.academy.getCode() + "-" + discipline + "-" + profile;
                        String right;
                        if ("IPR".equals(profile) || "IAN".equals(profile)) {
                            right = "Everything";
                        } else {
                            right = "Write";
                        }
                        
                        permissions.add(new Permission(user, right));
                    }
                }
                
                Document folder = this.createDocument(nuxeoSession, documentService, service, "UsageNumDossier", name, properties, true, permissions);

                folders.add(folder);
            } catch (Exception e) {
                // Do nothing
            }
        }

        return folders;
    }


    /**
     * Create activities.
     *
     * @param nuxeoSession Nuxeo session
     * @param documentService document service
     * @param folders parent folders
     * @param users activities referrers
     * @throws Exception
     */
    private void createActivities(Session nuxeoSession, DocumentService documentService, List<Document> folders, List<String> users) throws Exception {
        this.log.info("[" + this.academy.getLabel() + "] Création des activités");

        Fairy fairy = Fairy.create(this.locale);
        TextProducer textProducer = fairy.textProducer();
        Random random = new Random();

        int count = this.configuration.getActivitiesCount() / this.configuration.getFoldersCount();
        int progression = 1;
        int p;

        // CSV records
        List<CSVRecord> modalitesRecords = this.getRecords("unu_modalite");
        List<CSVRecord> disciplinesRecords = this.getRecords("disciplines");
        List<CSVRecord> naturesPedagogiquesRecords = this.getRecords("unu_naturepeda");
        List<CSVRecord> niveauxEducatifsRecords = this.getRecords("niveauxEducatifs");
        List<CSVRecord> lieuxRecords = this.getRecords("unu_lieu");
        List<CSVRecord> outilsRecords = this.getRecords("unu_outils_child1");
        List<CSVRecord> transferabiliteRecords = this.getRecords("unu_transferabilite");
        List<CSVRecord> pratiquesRecords = this.getRecords("unu_pratiques");
        List<CSVRecord> axesRecords = this.getRecords("carto_axes");
        List<CSVRecord> dureesRecords = this.getRecords("unu_duree");
        List<CSVRecord> niveauxRecords = this.getRecords("unu_niveau");
        List<CSVRecord> parcoursRecords = this.getRecords("carto_parcours");
        List<CSVRecord> niveauxReferentRecords = this.getRecords("unu_niveauReferent");

        for (Document folder : folders) {
            for (int i = 0; i < count; i++) {
                if ((progression % 10) == 0) {
                    this.log.info("[" + this.academy.getLabel() + "] Création des activités (" + String.valueOf(progression) + "/"
                            + String.valueOf(this.configuration.getActivitiesCount()) + ")");
                }
                progression++;

                // Vocabulaires
                float f = random.nextFloat();
                int v;
                if (f > 0.9) {
                    v = 3;
                } else if (f > 0.75) {
                    v = 2;
                } else {
                    v = 1;
                }
                Set<String> modalites = new HashSet<String>(v);
                Set<String> disciplines = new HashSet<String>(v);
                Set<String> naturesPedagogiques = new HashSet<String>(v);
                Set<String> niveauxEducatifs = new HashSet<String>(v);
                Set<String> lieux = new HashSet<String>(v);
                Set<String> outils = new HashSet<String>(v);
                Set<String> pratiques = new HashSet<String>(v);
                Set<String> axes = new HashSet<String>(v);
                Set<String> parcours = new HashSet<String>(v);
                for (int j = 0; j < v; j++) {
                    modalites.add(modalitesRecords.get(random.nextInt(modalitesRecords.size())).get("id"));
                    CSVRecord discipline = disciplinesRecords.get(random.nextInt(disciplinesRecords.size()));
                    disciplines.add(discipline.get("parent") + "/" + discipline.get("id"));
                    naturesPedagogiques.add(naturesPedagogiquesRecords.get(random.nextInt(naturesPedagogiquesRecords.size())).get("id"));
                    CSVRecord niveauEducatifRecord = niveauxEducatifsRecords.get(random.nextInt(niveauxEducatifsRecords.size()));
                    String niveauEducatif = niveauEducatifRecord.get("id");
                    while (niveauEducatifRecord != null) {
                        String parent = niveauEducatifRecord.get("parent");
                        if ((parent == null) || "__NULL__".equals(parent)) {
                            break;
                        }
                        
                        niveauEducatifRecord = null;
                        for (CSVRecord record : niveauxEducatifsRecords) {
                            String id = record.get("id");
                            if (parent.equals(id)) {
                                niveauEducatifRecord = record;
                                niveauEducatif = id + "/" + niveauEducatif;
                                break;
                            }
                        }
                    }
                    niveauxEducatifs.add(niveauEducatif);
                    lieux.add(lieuxRecords.get(random.nextInt(lieuxRecords.size())).get("id"));
                    CSVRecord outil = outilsRecords.get(random.nextInt(outilsRecords.size()));
                    outils.add(outil.get("parent") + "/" + outil.get("id"));
                    pratiques.add(pratiquesRecords.get(random.nextInt(pratiquesRecords.size())).get("id"));
                    axes.add(axesRecords.get(random.nextInt(axesRecords.size())).get("id"));
                    parcours.add(parcoursRecords.get(random.nextInt(parcoursRecords.size())).get("id"));
                }


                // Name
                String name = "act-" + String.valueOf(i + 1) + "-" + StringUtils.substringAfterLast(folder.getPath(), "/");

                // Properties
                PropertyMap properties = new PropertyMap();
                properties.set("dc:title", "Activité " + StringUtils.leftPad(String.valueOf(i + 1), 2, "0") + " (" + folder.getTitle() + ")");
                properties.set("acrp:activites",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:cadresUtilisations", StringUtils.join(modalites, SEPARATOR));
                properties.set("acrp:competences",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:disciplines", StringUtils.join(disciplines, SEPARATOR));
                properties.set("acrp:langues",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:naturesPedagogiques", StringUtils.join(naturesPedagogiques, SEPARATOR));
                properties.set("acrp:niveauxCompetences",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:niveauxEducatifs", StringUtils.join(niveauxEducatifs, SEPARATOR));
                properties.set("acrp:publicsCibles",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:sdCompetences",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:sdNiveauxCompetences",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:sdThemes",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:sousDomaines",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:themes",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("acrp:lieu", StringUtils.join(lieux, SEPARATOR));
                properties.set("acrp:outils", StringUtils.join(outils, SEPARATOR));
                properties.set("acrp:transferabilite", transferabiliteRecords.get(random.nextInt(transferabiliteRecords.size())).get("id"));
                properties.set("acrp:pratique", StringUtils.join(pratiques, SEPARATOR));
                properties.set("unum:analyse", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:axe", StringUtils.join(axes, SEPARATOR));
                properties.set("unum:duree", dureesRecords.get(random.nextInt(dureesRecords.size())).get("id"));
                properties.set("unum:experitheque", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:groupeViaeduc", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:logicielLocal",
                        StringUtils.join(StringUtils.split(textProducer.word(random.nextInt(3) + 1).replaceAll(",|;", StringUtils.EMPTY)), SEPARATOR));
                properties.set("unum:natPedaAutre", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:niv", niveauxRecords.get(random.nextInt(niveauxRecords.size())).get("id"));
                properties.set("unum:parcours", StringUtils.join(parcours, SEPARATOR));
                properties.set("unum:piege", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:resume", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:scenario", textProducer.sentence().replaceAll("\\r|\\n", StringUtils.EMPTY));
                properties.set("unum:urlExpe", fairy.company().url());

                // Internet softwares
                p = (random.nextInt(3) + 1);
                JSONArray internetSoftwares = new JSONArray();
                for (int j = 0; j < p; j++) {
                    Company company = fairy.company();

                    JSONObject software = new JSONObject();
                    software.put("name", company.name());
                    software.put("url", company.url());

                    internetSoftwares.add(software);
                }
                properties.set("unum:logicielInternet", internetSoftwares.toString());

                // Referrers
                Set<String> ids = new HashSet<String>(this.configuration.getReferrersCount());
                for (int j = 0; j < this.configuration.getReferrersCount(); j++) {
                    String user = users.get(random.nextInt(users.size()));
                    Attributes userAttributes = this.ldapContext.getAttributes(user);
                    String id = (String) userAttributes.get("uid").get();
                    ids.add(id.toString());
                }
                JSONArray referrers = new JSONArray();
                boolean owner = true;
                for (String id : ids) {
                    JSONObject referrer = new JSONObject();
                    referrer.put("uid", id);
                    referrer.put("niveau", niveauxReferentRecords.get(random.nextInt(niveauxReferentRecords.size())).get("id"));
                    referrer.put("owner", owner);

                    owner = false;

                    referrers.add(referrer);
                }
                properties.set("unum:referents", referrers.toString());

                // Application softwares
                JSONObject applicationSoftware = new JSONObject();
                JSONArray local = new JSONArray();
                local.add(fairy.company().name());
                applicationSoftware.put("local", local);
                p = (random.nextInt(3) + 1);
                JSONArray links = new JSONArray();
                for (int k = 0; k < p; k++) {
                    Person person = fairy.person();

                    JSONObject link = new JSONObject();
                    link.put("nom", person.fullName());
                    link.put("adresse", person.getAddress().toString());

                    links.add(link);
                }
                applicationSoftware.put("internet", links);
                properties.set("unula:logicielAppli", applicationSoftware.toString());

                try {
                    this.createDocument(nuxeoSession, documentService, folder, "UsageNum", name, properties, true);
                } catch (Exception e) {
                    // Do nothing
                }
            }
        }
    }


    /**
     * Create document if not exists.
     *
     * @param documentService document service
     * @param parent document parent
     * @param type document type
     * @param name document name
     * @param properties document properties
     * @return document
     * @throws Exception
     */
    private Document createDocument(Session nuxeoSession, DocumentService documentService, Document parent, String type, String name, PropertyMap properties)
            throws Exception {
        return this.createDocument(nuxeoSession, documentService, parent, type, name, properties, false);
    }


    /**
     * Create document if not exists.
     *
     * @param documentService document service
     * @param parent document parent
     * @param type document type
     * @param name document name
     * @param properties document properties
     * @param permissions permissions
     * @return document
     * @throws Exception
     */
    private Document createDocument(Session nuxeoSession, DocumentService documentService, Document parent, String type, String name, PropertyMap properties,
            List<Permission> permissions) throws Exception {
        return this.createDocument(nuxeoSession, documentService, parent, type, name, properties, false, permissions);
    }


    /**
     * Create document if not exists.
     *
     * @param documentService document service
     * @param parent document parent
     * @param type document type
     * @param name document name
     * @param properties document properties
     * @param publish publish indicator
     * @return document
     * @throws Exception
     */
    private Document createDocument(Session nuxeoSession, DocumentService documentService, Document parent, String type, String name, PropertyMap properties,
            boolean publish) throws Exception {
        return this.createDocument(nuxeoSession, documentService, parent, type, name, properties, false, null);
    }


    /**
     * Create document if not exists.
     *
     * @param documentService document service
     * @param parent document parent
     * @param type document type
     * @param name document name
     * @param properties document properties
     * @param publish publish indicator
     * @param permissions permissions
     * @return document
     * @throws Exception
     */
    private Document createDocument(Session nuxeoSession, DocumentService documentService, Document parent, String type, String name, PropertyMap properties,
            boolean publish, List<Permission> permissions) throws Exception {
        Document document;
        try {
            String path = StringUtils.removeEnd(parent.getPath(), "/") + "/" + name;
            document = documentService.getDocument(new PathRef(path));
        } catch (Exception e1) {
            try {
                document = documentService.createDocument(parent, type, name, properties);
                if (permissions != null) {
                    for (Permission permission : permissions) {
                        document = documentService.setPermission(document, permission.user, permission.right);
                    }
                }
                if (publish) {
                    OperationRequest request = nuxeoSession.newRequest("setOnLine").setInput(document);
                    request.execute();
                }
            } catch (Exception e2) {
                this.log.error("[" + this.academy.getLabel() + "] Erreur de création avec les données suivantes :");
                this.log.error("   name = '" + name + "'");
                this.log.error("   properties = '" + properties + "'");

                this.errors++;

                throw e2;
            }
        }

        // WebId
        String webId = properties.getString("ttc:webid");
        if (StringUtils.isNotEmpty(webId)) {
            document.set("ttc:webid", webId);
        }

        return document;
    }


    /**
     * Get CSV records.
     *
     * @param name CSV file name
     * @return CSV records
     * @throws IOException
     */
    private List<CSVRecord> getRecords(String name) throws IOException {
        CSVParser parser = null;
        List<CSVRecord> records = null;
        try {
            InputStream in = this.getClass().getClassLoader().getResourceAsStream("csv/" + name + ".csv");
            Reader reader = new InputStreamReader(in);
            parser = new CSVParser(reader, CSVFormat.DEFAULT.withHeader());
            records = parser.getRecords();
        } finally {
            parser.close();
        }
        return records;
    }


    /**
     * Generate geolocation.
     * 
     * @param random random generator
     * @return geolocation
     */
    private String generateGeolocation(Random random) {
        double latitude = random.nextDouble() * 9d + 42d;
        double longitude = random.nextDouble() * 12d - 4d;
        return String.valueOf(latitude) + SEPARATOR + String.valueOf(longitude);
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getId() {
        return this.academy.getCode();
    }


    /**
     * Permission inner-class.
     * 
     * @author Cédric Krommenhoek
     */
    private class Permission {

        /** User or group name. */
        private final String user;
        /** Right. */
        private final String right;


        /**
         * Constructor.
         * 
         * @param user user or group name
         * @param right right
         */
        public Permission(String user, String right) {
            super();
            this.user = user;
            this.right = right;
        }

    }

}
