/*
 * (C) Copyright 2017 OSIVIA (http://www.osivia.com) 
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 */
package fr.toutatice.services.calendar.batch;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.PortletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.Documents;
import org.nuxeo.ecm.automation.client.model.PropertyList;
import org.nuxeo.ecm.automation.client.model.PropertyMap;
import org.osivia.portal.api.batch.AbstractBatch;
import org.osivia.portal.api.cache.services.CacheInfo;
import org.osivia.services.calendar.edition.portlet.model.CalendarSynchronizationSource;
import org.osivia.services.calendar.view.portlet.model.events.EventKey;
import org.osivia.services.calendar.view.portlet.model.events.EventToSync;

import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoQueryFilterContext;
import fr.toutatice.portail.cms.nuxeo.api.services.NuxeoCommandContext;
import fr.toutatice.services.calendar.batch.command.InteractikSynchronizationCommand;
import fr.toutatice.services.calendar.batch.command.ListInteractikAgendaCommand;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikEventToSync;
import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.data.CalendarParser;
import net.fortuna.ical4j.data.CalendarParserFactory;
import net.fortuna.ical4j.data.ParserException;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.ComponentList;
import net.fortuna.ical4j.model.ParameterFactoryRegistry;
import net.fortuna.ical4j.model.PropertyFactoryRegistry;
import net.fortuna.ical4j.model.TimeZone;
import net.fortuna.ical4j.model.TimeZoneRegistry;
import net.fortuna.ical4j.model.TimeZoneRegistryFactory;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.component.VTimeZone;

/**
 * Batch de synchronisation des événements Evrigo
 * Analyse des calendriers interactik
 * Pour chaque source de synchronisation de ces calendriers, on synchronise les événements
 * 
 * @author Julien Barberet
 *
 */
public class SynchronizationEvrigoBatch extends AbstractBatch {

	/** List of synchronization sources */
	private static final String LIST_SOURCE_SYNCHRO = "cal:sources";
	/** URL to synchronize agenda */
	private static final String URL_SYNCHRONIZATION = "url";
	/** Source ID */
	private static final String SOURCEID_SYNCHRONIZATION = "sourceId";
	
	private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
	
	private static final String DEPARTEMENT_PROPERTY = "X-DEPARTEMENT";
	
	private static final String VILLE_PROPERTY = "X-VILLE";
	
	private static final String DATE_DEBUT_INSCRIPTION_PROPERTY = "X-DATE_DEBUT_INSCRIPTION";
	
	private static final String DATE_FIN_INSCRIPTION_PROPERTY = "X-DATE_FIN_INSCRIPTION";
	
	private static final String URLINSCRIPTION_PROPERTY = "X-URLINSCRIPTION";

	/** Portlet context. */
	private static PortletContext portletContext;

	private static final String CRON_DEFAULT_VALUE = "0 0/2 * * * ?";

	/** logger */
	protected static final Log logger = LogFactory.getLog(SynchronizationEvrigoBatch.class);

	public SynchronizationEvrigoBatch() {
	}

	@Override
	public boolean isRunningOnMasterOnly() {
		return true;
	}

	@Override
	public String getJobScheduling() {
		return System.getProperty("CRON_BATCH_EVRIGO", CRON_DEFAULT_VALUE);
	}

	@Override
	public void execute(Map<String, Object> parameters) {
		logger.info("Exécution du batch import Evrigo");

		// 1) list interactik agendas that have synchronization sources
		NuxeoController nuxeoController = new NuxeoController(portletContext);
		nuxeoController.setAuthType(NuxeoCommandContext.AUTH_TYPE_SUPERUSER);
		nuxeoController.setCacheType(CacheInfo.CACHE_SCOPE_PORTLET_CONTEXT);

		INuxeoCommand nuxeoCommand = new ListInteractikAgendaCommand(NuxeoQueryFilterContext.CONTEXT_LIVE_N_PUBLISHED);
		Documents agendasInteractik = (Documents) nuxeoController.executeNuxeoCommand(nuxeoCommand);

		// 2) For each interactik agenda, search synchronization source
		for (Document agenda : agendasInteractik) {
			PropertyList propertyList = (PropertyList) agenda.getProperties().get(LIST_SOURCE_SYNCHRO);
			if (propertyList != null) {
				CalendarSynchronizationSource source;
				for (int i = 0; i < propertyList.size(); i++) {
					PropertyMap map = propertyList.getMap(i);
					source = new CalendarSynchronizationSource();
					source.setUrl(map.getString(URL_SYNCHRONIZATION));
					source.setId(map.getString(SOURCEID_SYNCHRONIZATION));
					// 3) Synchronize each synchronization source
					try {
						Map<EventKey, EventToSync> mapEvents = getEventsFromUrl(new URL(source.getUrl()), source.getId());
						nuxeoCommand = new InteractikSynchronizationCommand(NuxeoQueryFilterContext.CONTEXT_LIVE_N_PUBLISHED,agenda.getPath(), mapEvents);
						nuxeoController.executeNuxeoCommand(nuxeoCommand);
					} catch (MalformedURLException e) {
						logger.error("Impossible d'accéder à l'URL de synchronisation, détail:" + e.getMessage());
					}
				}
			}
		}

	}

	public void setPortletContext(PortletContext portletContext) {
		SynchronizationEvrigoBatch.portletContext = portletContext;
	}

	/**
	 * Events from the URL parameter
	 * @param url url to synchronize
	 * @param idSource
	 * @return
	 */
	private Map<EventKey, EventToSync> getEventsFromUrl(URL url, String idSource) {

		HashMap<EventKey, EventToSync> mapEvents = new HashMap<EventKey, EventToSync>();
		try {
			URLConnection conn = url.openConnection();

			CalendarParser parser = CalendarParserFactory.getInstance().createParser();

			PropertyFactoryRegistry propertyFactoryRegistry = new PropertyFactoryRegistry();

			ParameterFactoryRegistry parameterFactoryRegistry = new ParameterFactoryRegistry();

			TimeZoneRegistry tzRegistry = TimeZoneRegistryFactory.getInstance().createRegistry();

			CalendarBuilder builder = new CalendarBuilder(parser, propertyFactoryRegistry, parameterFactoryRegistry,
					tzRegistry);
			BufferedReader rd = new BufferedReader(new InputStreamReader(conn.getInputStream()));

			net.fortuna.ical4j.model.Calendar calendar;
			List<VEvent> listEvent = new ArrayList<VEvent>();

			calendar = builder.build(rd);
			ComponentList listComponent = calendar.getComponents();
			listEvent = listComponent.getComponents(Component.VEVENT);
			VTimeZone vTimeZoneAllEvent = (VTimeZone) calendar.getComponent(Component.VTIMEZONE);
			TimeZone timeZoneAllEvent = null;

			// If timezone not inquire, default timezone is GMT
			if (vTimeZoneAllEvent == null) {
				TimeZoneRegistry registry = TimeZoneRegistryFactory.getInstance().createRegistry();
				timeZoneAllEvent = registry.getTimeZone("GMT");
			} else {
				timeZoneAllEvent = new TimeZone(vTimeZoneAllEvent);
			}

			Calendar cal;
			EventKey key;
			for (VEvent event : listEvent) {
				if (event.getRecurrenceId() == null) {
					cal = null;
				} else {
					cal = Calendar.getInstance();
					if (event.getRecurrenceId().getTimeZone() == null) {
						cal.setTimeZone(timeZoneAllEvent);
					} else {
						cal.setTimeZone(event.getRecurrenceId().getTimeZone());
					}
					cal.setTime(event.getRecurrenceId().getDate());
				}
				key = new EventKey(event.getUid().getValue(), idSource, cal);
				mapEvents.put(key, buildEvent(event, idSource, timeZoneAllEvent, url.toString()));
			}

		} catch (ParserException e) {
			logger.error("Erreur de parsing lors de la synchronisation, détail:" + e.getMessage());
		} catch (MalformedURLException e) {
			logger.error("Impossible d'accéder à l'URL de synchronisation, détail:"+e.getMessage());
		} catch (IOException e) {
			logger.error("Erreur de parsing lors de la synchronisation, détail:"+e.getMessage());
		}
		logger.warn(mapEvents.size()+" événements parsés avec succès");
		return mapEvents;
	}

	/**
	 * Build InteractikEventToSync from iCal event
	 * @param vevent
	 * @param idAgenda
	 * @param timeZoneAllEvent
	 * @param url
	 * @return
	 */
	private InteractikEventToSync buildEvent(VEvent vevent, String idAgenda, TimeZone timeZoneAllEvent, String url) {
		boolean allDay = (vevent.getStartDate().getValue().length() == 8
				&& vevent.getEndDate().getValue().length() == 8);

		Calendar startCal = Calendar.getInstance();
		if (vevent.getStartDate().getTimeZone() != null) {
			startCal.setTimeZone(vevent.getStartDate().getTimeZone());
		} else {
			startCal.setTimeZone(timeZoneAllEvent);
		}
		startCal.setTime(vevent.getStartDate().getDate());

		Calendar endCal = Calendar.getInstance();
		if (vevent.getEndDate().getTimeZone() != null) {
			endCal.setTimeZone(vevent.getEndDate().getTimeZone());
		} else {
			endCal.setTimeZone(timeZoneAllEvent);
		}
		endCal.setTime(vevent.getEndDate().getDate());

		Calendar createdCal = Calendar.getInstance();
		if (vevent.getCreated().getTimeZone() != null) {
			createdCal.setTimeZone(vevent.getCreated().getTimeZone());
		} else {
			createdCal.setTimeZone(timeZoneAllEvent);
		}
		createdCal.setTime(vevent.getCreated().getDate());

		Calendar lastModifiedCal = Calendar.getInstance();
		if (vevent.getLastModified().getTimeZone() != null) {
			lastModifiedCal.setTimeZone(vevent.getLastModified().getTimeZone());
		} else {
			lastModifiedCal.setTimeZone(timeZoneAllEvent);
		}
		lastModifiedCal.setTime(vevent.getLastModified().getDate());

		Calendar startReccuringStartSource;
		if (vevent.getRecurrenceId() == null) {
			startReccuringStartSource = null;
		} else {
			startReccuringStartSource = Calendar.getInstance();
			if (vevent.getRecurrenceId().getTimeZone() != null) {
				startReccuringStartSource.setTimeZone(vevent.getRecurrenceId().getTimeZone());
			} else {
				startReccuringStartSource.setTimeZone(timeZoneAllEvent);
			}
			startReccuringStartSource.setTime(vevent.getRecurrenceId().getDate());
		}
		String description = (vevent.getDescription() == null) ? null : vevent.getDescription().getValue();
		String summary = (vevent.getSummary() == null) ? null : vevent.getSummary().getValue();
		String uid = (vevent.getUid() == null) ? null : vevent.getUid().getValue();
		String location = (vevent.getLocation() == null) ? null : vevent.getLocation().getValue();
		
		String departement = vevent.getProperty(DEPARTEMENT_PROPERTY) != null? vevent.getProperty(DEPARTEMENT_PROPERTY).getValue() : "";
		String ville = vevent.getProperty(VILLE_PROPERTY) != null? vevent.getProperty(VILLE_PROPERTY).getValue() : "";
		String urlInscription = vevent.getProperty(URLINSCRIPTION_PROPERTY) != null? vevent.getProperty(URLINSCRIPTION_PROPERTY).getValue() : "";
		String dateDebutInscription = vevent.getProperty(DATE_DEBUT_INSCRIPTION_PROPERTY) != null? vevent.getProperty(DATE_DEBUT_INSCRIPTION_PROPERTY).getValue() : "";
		String dateFinInscription = vevent.getProperty(DATE_FIN_INSCRIPTION_PROPERTY) != null? vevent.getProperty(DATE_FIN_INSCRIPTION_PROPERTY).getValue() : "";

		
		Date debutInscription = null;
		try {
			debutInscription = dateDebutInscription != null? sdf.parse(dateDebutInscription) : null;
		} catch (ParseException e) {
			logger.error("Erreur de parsing de la date de debut d'inscription :" + dateDebutInscription + " avec le format "+sdf.toPattern()
			+"\n URL de synchro : "+url
			+"\n UID : "+uid);
		}
		
		Date finInscription = null;
		try {
			finInscription = sdf.parse(dateFinInscription);
		} catch (ParseException e) {
			logger.error("Erreur de parsing de la date de fin d'inscription :" + dateFinInscription + " avec le format "+sdf.toPattern()
			+"\n URL de synchro : "+url
			+"\n UID : "+uid);
		}
		
		return new InteractikEventToSync(null, summary, allDay, startCal, endCal, description, idAgenda, uid, createdCal,
				lastModifiedCal, startReccuringStartSource, location, departement, ville, debutInscription, finInscription, urlInscription);
	}
}
