package fr.toutatice.services.calendar.view.portlet.service;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;

import javax.portlet.PortletException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.services.calendar.view.portlet.model.events.Event;
import org.osivia.services.calendar.view.portlet.service.CalendarIntegrationService;
import org.osivia.services.calendar.view.portlet.service.CalendarIntegrationServiceImpl;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import fr.toutatice.services.calendar.batch.SynchronizationEvrigoBatch;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikEvent;
import net.fortuna.ical4j.model.PropertyList;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.XProperty;

/**
 * Interactik calendar integration portlet service implementation.
 * 
 * @author Cédric Krommenhoek
 * @see CalendarIntegrationServiceImpl
 * @see CalendarIntegrationService
 */
@Service
@Primary
public class InteractikCalendarIntegrationServiceImpl extends CalendarIntegrationServiceImpl implements CalendarIntegrationService {

    /** Inscription date format. */
    private final DateFormat inscriptionDateFormat;


    /**
     * Constructor.
     */
    public InteractikCalendarIntegrationServiceImpl() {
        super();

        // Inscription date format
        this.inscriptionDateFormat = new SimpleDateFormat("yyyyMMdd");
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected VEvent createVEvent(PortalControllerContext portalControllerContext, Event event) throws PortletException, IOException {
        VEvent vevent = super.createVEvent(portalControllerContext, event);
        PropertyList properties = vevent.getProperties();

        if (event instanceof InteractikEvent) {
            InteractikEvent interactikEvent = (InteractikEvent) event;

            // Département
            if (StringUtils.isNotEmpty(interactikEvent.getDepartement())) {
                XProperty departement = new XProperty(SynchronizationEvrigoBatch.DEPARTEMENT_PROPERTY, interactikEvent.getDepartement());
                properties.add(departement);
            }

            // Ville
            if (StringUtils.isNotEmpty(interactikEvent.getVille())) {
                XProperty ville = new XProperty(SynchronizationEvrigoBatch.VILLE_PROPERTY, interactikEvent.getVille());
                properties.add(ville);
            }

            // URL d'inscription
            if (StringUtils.isNotEmpty(interactikEvent.getUrlInscription())) {
                XProperty urlInscription = new XProperty(SynchronizationEvrigoBatch.URLINSCRIPTION_PROPERTY, interactikEvent.getUrlInscription());
                properties.add(urlInscription);
            }

            // Date de début d'inscription
            if (interactikEvent.getDebutInscription() != null) {
                XProperty debutInscription = new XProperty(SynchronizationEvrigoBatch.DATE_DEBUT_INSCRIPTION_PROPERTY,
                        this.inscriptionDateFormat.format(interactikEvent.getDebutInscription()));
                properties.add(debutInscription);
            }

            // Date de fin d'inscription
            if (interactikEvent.getFinInscription() != null) {
                XProperty finInscription = new XProperty(SynchronizationEvrigoBatch.DATE_FIN_INSCRIPTION_PROPERTY,
                        this.inscriptionDateFormat.format(interactikEvent.getFinInscription()));
                properties.add(finInscription);
            }
            
            // Visuel de l'événement
//            if (interactikEvent.getVisuelEvt() != null) {
//                XProperty visuelEvt = new XProperty(SynchronizationEvrigoBatch.VISUEL_EVT_PROPERTY, interactikEvent.getVisuelEvt());
//                properties.add(visuelEvt);
//            }    
            
            // Organisateurs
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.ORGANISATEURS_PROPERTY, interactikEvent.getOrganisateurs());

            // Pôles disciplinaires
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.POLES_DISCIPLINAIRES_PROPERTY, interactikEvent.getPolesDisciplinaires());

            // Thèmes
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.THEMES_PROPERTY, interactikEvent.getThemes());

            // Niveaux
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.NIVEAUX_PROPERTY, interactikEvent.getNiveaux());

            // Contenus
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.CONTENUS_PROPERTY, interactikEvent.getContenus());

            // Repères DNE
            this.addMultipleProperties(properties, SynchronizationEvrigoBatch.REPERES_DNE_PROPERTY, interactikEvent.getReperesDne());
        }

        return vevent;
    }


    /**
     * Add multiple properties.
     * 
     * @param properties properties
     * @param name property name
     * @param values property values
     */
    private void addMultipleProperties(PropertyList properties, String name, List<String> values) {
        if (CollectionUtils.isNotEmpty(values)) {
            for (String value : values) {
                XProperty property = new XProperty(name, value);
                properties.add(property);
            }
        }
    }

}
