package fr.toutatice.services.calendar.view.portlet.service;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.services.calendar.common.model.InteractikCalendarColor;
import fr.toutatice.services.calendar.view.portlet.model.Cooperative;
import fr.toutatice.services.calendar.view.portlet.model.InteractikFilters;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikDailyCalendarEventsData;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikDailyEvent;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikEventToSync;
import net.fortuna.ical4j.model.TimeZone;
import net.fortuna.ical4j.model.component.VEvent;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.jboss.portal.theme.impl.render.dynamic.DynaRenderOptions;
import org.osivia.portal.api.Constants;
import org.osivia.portal.api.PortalException;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.internationalization.Bundle;
import org.osivia.portal.api.internationalization.IBundleFactory;
import org.osivia.portal.api.menubar.MenubarGroup;
import org.osivia.portal.api.menubar.MenubarItem;
import org.osivia.portal.api.urls.IPortalUrlFactory;
import org.osivia.services.calendar.common.model.CalendarColor;
import org.osivia.services.calendar.common.service.CalendarServiceImpl;
import org.osivia.services.calendar.edition.portlet.service.CalendarEditionService;
import org.osivia.services.calendar.view.portlet.model.CalendarOptions;
import org.osivia.services.calendar.view.portlet.model.calendar.CalendarData;
import org.osivia.services.calendar.view.portlet.model.events.EventsData;
import org.osivia.services.calendar.view.portlet.service.CalendarViewService;
import org.osivia.services.calendar.view.portlet.service.CalendarViewServiceImpl;
import org.osivia.services.calendar.view.portlet.service.generator.ICalendarGenerator;
import org.osivia.services.calendar.view.portlet.utils.PeriodTypes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.*;

/**
 * Calendar service implementation.
 *
 * @author Cédric Krommenhoek
 * @see CalendarServiceImpl
 * @see CalendarViewService
 */
@Service
@Primary
public class InteractikCalendarViewServiceImpl extends CalendarViewServiceImpl implements InteractikCalendarViewService {

	private static final String DEPARTEMENT_PROPERTY = "X-DEPARTEMENT";

	private static final String VILLE_PROPERTY = "X-VILLE";

	private static final String DATE_DEBUT_INSCRIPTION_PROPERTY = "X-DATE_DEBUT_INSCRIPTION";

	private static final String DATE_FIN_INSCRIPTION_PROPERTY = "X-DATE_FIN_INSCRIPTION";

	private static final String URLINSCRIPTION_PROPERTY = "X-URLINSCRIPTION";

	private static final String DATE_FORMAT = "yyyyMMdd";


    /**
     * Portal URL factory.
     */
    @Autowired
    private IPortalUrlFactory portalUrlFactory;
    /**
     * Internationalization bundle factory.
     */
    @Autowired
    private IBundleFactory bundleFactory;


    /**
     * Constructor.
     */
    public InteractikCalendarViewServiceImpl() {
        super();
    }


    @Override
    protected void addIntegrationMenubarItem(PortalControllerContext portalControllerContext) throws PortletException {
        super.addIntegrationMenubarItem(portalControllerContext);

        // Calendar options
        CalendarOptions options = this.repository.getConfiguration(portalControllerContext);

        if (options.isIntegration()) {
            // Portlet request
            PortletRequest request = portalControllerContext.getRequest();
            // Internationalization bundle
            Bundle bundle = this.bundleFactory.getBundle(request.getLocale());
            // Menubar
            List<MenubarItem> menubar = (List<MenubarItem>) request.getAttribute(Constants.PORTLET_ATTR_MENU_BAR);

            // Calendar path
            String path = this.repository.getCalendarPath(portalControllerContext);

            // Menubar item order
            int order = 1;

            for (Cooperative cooperative : Cooperative.values()) {
                // URL
                String url;
                if (StringUtils.isEmpty(path)) {
                    url = null;
                } else {
                    // Filters
                    List<String> filters = new ArrayList<>();
                    filters.add(StringEscapeUtils.escapeHtml("agenda=" + cooperative.name()));

                    // Parameters
                    Map<String, String> parameters = new HashMap<>();
                    parameters.put("format", "ics");
                    parameters.put("filters", StringUtils.join(filters, "&"));

                    try {
                        url = this.portalUrlFactory.getPermaLink(portalControllerContext, "integration", parameters, path,
                                IPortalUrlFactory.PERM_LINK_TYPE_PORTLET_RESOURCE);
                    } catch (PortalException e) {
                        throw new PortletException(e);
                    }
                }

                if (StringUtils.isNotEmpty(url)) {
                    // Title
                    String arg = bundle.getString(cooperative.getKey());
                    String title = bundle.getString("TIK_CALENDAR_INTEGRATION_ICS_MENUBAR_ITEM", arg);

                    MenubarItem item = new MenubarItem("CALENDAR_INTEGRATION_ICS", title, null, MenubarGroup.SPECIFIC, order++, url, null, null, null);
                    menubar.add(item);
                }
            }
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public JSONArray loadEventsArray(PortalControllerContext portalControllerContext, CalendarData calendarData) throws PortletException {
        EventsData eventsData = getEventsData(portalControllerContext, calendarData);
        List<InteractikDailyEvent> listEvent = ((InteractikDailyCalendarEventsData) eventsData).getEvents();

        Map<String, CalendarColor> mapColor = getSourcesColor(portalControllerContext);

        JSONArray array = new JSONArray();
        JSONObject object = null;
        Iterator<InteractikDailyEvent> iterator = listEvent.iterator();
        InteractikDailyEvent event = null;
        SimpleDateFormat formater = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        SimpleDateFormat formaterAllDay = new SimpleDateFormat("yyyy-MM-dd");

        while (iterator.hasNext()) {
            event = iterator.next();

            object = new JSONObject();

            object.put("text", event.getTitle());
            if (event.isAllDay()) {
                object.put("start_date", formaterAllDay.format(event.getStartDate()));
                object.put("end_date", formaterAllDay.format(event.getEndDate()));
            } else {
                object.put("start_date", formater.format(event.getStartDate()));
                object.put("end_date", formater.format(event.getEndDate()));
            }
            object.put("doc_id", event.getId());
            object.put("color", event.getBckgColor());
            object.put("view_url", event.getViewURL());
            if (event.getBckgColor() == null) {
                if (event.getIdParentSource() == null) {
                    if (mapColor.get("PRIMARY") != null) {
                        object.put("extraClass", mapColor.get("PRIMARY").getBackgroundClass());
                    }
                } else {
                    if (mapColor.get(event.getIdParentSource()) == null) {
                        object.put("extraClass", InteractikCalendarColor.DEFAULT);
                    } else {
                        object.put("extraClass", mapColor.get(event.getIdParentSource()).getBackgroundClass());
                    }
                }
            } else {
                object.put("extraClass", InteractikCalendarColor.fromId(event.getBckgColor()).getBackgroundClass());
            }
            object.put("ville", event.getVille());
            object.put("readonly", this.isEventReadOnly(portalControllerContext, event));

            array.add(object);
        }
        return array;
    }

    /**
     * {@inheritDoc}
     * @throws PortletException
     */
    @Override
    public InteractikEventToSync buildEvent(VEvent vevent, String idAgenda, TimeZone timeZoneAllEvent) throws PortletException {
        boolean allDay = (vevent.getStartDate().getValue().length() == 8 && vevent.getEndDate().getValue().length() == 8);

        Calendar startCal = Calendar.getInstance();
        if (vevent.getStartDate().getTimeZone() != null) {
            startCal.setTimeZone(vevent.getStartDate().getTimeZone());
        } else {
            startCal.setTimeZone(timeZoneAllEvent);
        }
        startCal.setTime(vevent.getStartDate().getDate());

        Calendar endCal = Calendar.getInstance();
        if (vevent.getEndDate().getTimeZone() != null) {
            endCal.setTimeZone(vevent.getEndDate().getTimeZone());
        } else {
            endCal.setTimeZone(timeZoneAllEvent);
        }
        endCal.setTime(vevent.getEndDate().getDate());

        Calendar createdCal = Calendar.getInstance();
        if (vevent.getCreated().getTimeZone() != null) {
            createdCal.setTimeZone(vevent.getCreated().getTimeZone());
        } else {
            createdCal.setTimeZone(timeZoneAllEvent);
        }
        createdCal.setTime(vevent.getCreated().getDate());

        Calendar lastModifiedCal = Calendar.getInstance();
        if (vevent.getLastModified().getTimeZone() != null) {
            lastModifiedCal.setTimeZone(vevent.getLastModified().getTimeZone());
        } else {
            lastModifiedCal.setTimeZone(timeZoneAllEvent);
        }
        lastModifiedCal.setTime(vevent.getLastModified().getDate());

        Calendar startReccuringStartSource;
        if (vevent.getRecurrenceId() == null) {
            startReccuringStartSource = null;
        } else {
            startReccuringStartSource = Calendar.getInstance();
            if (vevent.getRecurrenceId().getTimeZone() != null) {
                startReccuringStartSource.setTimeZone(vevent.getRecurrenceId().getTimeZone());
            } else {
                startReccuringStartSource.setTimeZone(timeZoneAllEvent);
            }
            startReccuringStartSource.setTime(vevent.getRecurrenceId().getDate());
        }
        String description = (vevent.getDescription() == null) ? null : vevent.getDescription().getValue();
        String summary = (vevent.getSummary() == null) ? null : vevent.getSummary().getValue();
        String uid = (vevent.getUid() == null) ? null : vevent.getUid().getValue();
        String location = (vevent.getLocation() == null) ? null : vevent.getLocation().getValue();

        String department = null;
        String urlInscription = null;
        String ville = null;
        Date dateDebutInscription = null;
        Date dateFinInscription = null;
        if (vevent.getProperty(DEPARTEMENT_PROPERTY) != null) department = vevent.getProperty(DEPARTEMENT_PROPERTY).getValue();
        if (vevent.getProperty(URLINSCRIPTION_PROPERTY) != null) urlInscription = vevent.getProperty(URLINSCRIPTION_PROPERTY).getValue();
        if (vevent.getProperty(VILLE_PROPERTY) != null) ville = vevent.getProperty(VILLE_PROPERTY).getValue();
        DateFormat format = new SimpleDateFormat(DATE_FORMAT);
        if (vevent.getProperty(DATE_DEBUT_INSCRIPTION_PROPERTY) != null)
        {
        	String date = vevent.getProperty(DATE_DEBUT_INSCRIPTION_PROPERTY).getValue();
        	try {
				dateDebutInscription = format.parse(date);
			} catch (ParseException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
        }

        if (vevent.getProperty(DATE_FIN_INSCRIPTION_PROPERTY) != null)
        {
        	String date = vevent.getProperty(DATE_FIN_INSCRIPTION_PROPERTY).getValue();
        	try {
				dateFinInscription = format.parse(date);
			} catch (ParseException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
        }

        return new InteractikEventToSync(null, summary, allDay, startCal, endCal, description, idAgenda, uid, createdCal, lastModifiedCal, startReccuringStartSource, location, department, ville, dateDebutInscription, dateFinInscription, urlInscription);
    }

    /**
     * Get generator from his period type.
     *
     * @param portalControllerContext portal controller context
     * @param periodType period type
     * @return generator
     * @throws PortletException
     */
    private ICalendarGenerator getGenerator(PortalControllerContext portalControllerContext, PeriodTypes periodType) throws PortletException {
        ICalendarGenerator result = null;

        // Search generator into application context
        Map<String, ICalendarGenerator> generators = this.applicationContext.getBeansOfType(ICalendarGenerator.class);
        for (ICalendarGenerator generator : generators.values()) {
            if (generator.getPeriodType().getViewPath().equals(periodType.getViewPath())) {
                result = generator;
                generator.setPeriodType(periodType);
                break;
            }
        }

        return result;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getEventCreationRedirectionUrl(PortalControllerContext portalControllerContext) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);

        // Calendar options
        CalendarOptions options = this.repository.getConfiguration(portalControllerContext);

        // Portlet instance
        String instance = "interactik-calendar-event-edition-instance";

        // Path
        String path = nuxeoController.getComputedPath(options.getCmsPath());

        // Window properties
        Map<String, String> properties = new HashMap<>();
        properties.put(Constants.WINDOW_PROP_URI, path);
        properties.put(DynaRenderOptions.PARTIAL_REFRESH_ENABLED, String.valueOf(true));
        properties.put("osivia.ajaxLink", "1");
        properties.put(CalendarEditionService.CREATION_PROPERTY, String.valueOf(true));

        // Redirection URL
        String redirectionUrl;
        try {
            redirectionUrl = this.portalUrlFactory.getStartPortletUrl(portalControllerContext, instance, properties);
        } catch (PortalException e) {
            throw new PortletException(e);
        }

        return redirectionUrl;
    }


	@Override
	public InteractikFilters getFilters(PortalControllerContext portalControllerContext) throws PortletException {

		PortletRequest request = portalControllerContext.getRequest();

        InteractikFilters filters = this.applicationContext.getBean(InteractikFilters.class);

		filters.setCooperative22(BooleanUtils.toBooleanDefaultIfNull(BooleanUtils.toBooleanObject(request.getParameter(COOPERATIVE_22)), true));
		filters.setCooperative29(BooleanUtils.toBooleanDefaultIfNull(BooleanUtils.toBooleanObject(request.getParameter(COOPERATIVE_29)), true));
		filters.setCooperative35(BooleanUtils.toBooleanDefaultIfNull(BooleanUtils.toBooleanObject(request.getParameter(COOPERATIVE_35)), true));
		filters.setCooperative56(BooleanUtils.toBooleanDefaultIfNull(BooleanUtils.toBooleanObject(request.getParameter(COOPERATIVE_56)), true));
		filters.setCooperativeAcademie(BooleanUtils.toBooleanDefaultIfNull(BooleanUtils.toBooleanObject(request.getParameter(COOPERATIVE_ACADEMIE)), true));

		return filters;
	}

}

