package fr.toutatice.services.calendar.event.view.portlet.service;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.CODE_DEPARTEMENT_ACADEMIE;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DEPARTEMENT_PROPERTY;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import javax.portlet.PortletException;
import javax.portlet.PortletRequest;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.http.client.utils.URIBuilder;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.PropertyMap;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.internationalization.Bundle;
import org.osivia.portal.api.internationalization.IBundleFactory;
import org.osivia.portal.api.urls.IPortalUrlFactory;
import org.osivia.portal.core.web.IWebIdService;
import org.osivia.services.calendar.common.model.CalendarColor;
import org.osivia.services.calendar.event.view.portlet.service.CalendarEventViewServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.cms.NuxeoDocumentContext;
import fr.toutatice.services.calendar.common.model.InteractikCalendarColor;
import fr.toutatice.services.calendar.common.repository.InteractikCalendarEventRepository;
import fr.toutatice.services.calendar.event.view.portlet.model.InteractikCalendarEventViewForm;

/**
 * Interactik calendar event view portlet service implementation.
 *
 * @author Julien Barberet
 * @author Cédric Krommenhoek
 * @see CalendarEventViewServiceImpl
 * @see InteractikCalendarEventViewService
 */
@Service
@Primary
public class InteractikCalendarEventViewServiceImpl extends CalendarEventViewServiceImpl implements InteractikCalendarEventViewService {
	
    /** Common repository. */
    @Autowired
    private InteractikCalendarEventRepository commonRepository;

    /** Portal URL factory. */
    @Autowired
    private IPortalUrlFactory portalUrlFactory;

    /** WebId service. */
    @Autowired
    private IWebIdService webIdService;

    /** Internationalization bundle factory. */
    @Autowired
    private IBundleFactory bundleFactory;


    /** Facebook application identifier. */
    private final String facebookAppId;


    /**
     * Constructor.
     */
    public InteractikCalendarEventViewServiceImpl() {
        super();
        this.facebookAppId = System.getProperty("acrennes.sites.interactik.facebook.app.id");
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getBackUrl(PortalControllerContext portalControllerContext) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);

        // Navigation path
        String navigationPath = nuxeoController.getNavigationPath();

        // CMS path
        String cmsPath;
        if (StringUtils.isEmpty(navigationPath)) {
            cmsPath = null;
        } else {
            // Nuxeo document context
            NuxeoDocumentContext documentContext = nuxeoController.getDocumentContext(navigationPath);
            // WebId
            String webId = documentContext.getWebId();

            cmsPath = this.webIdService.webIdToCmsPath(webId);
        }

        // URL
        String url;
        if (StringUtils.isEmpty(cmsPath)) {
            url = null;
        } else {
            url = this.portalUrlFactory.getCMSUrl(portalControllerContext, null, cmsPath, null, null, null, null, null, null, null);
        }

        return url;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public InteractikCalendarEventViewForm getForm(PortalControllerContext portalControllerContext) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);
        // Portlet request
        PortletRequest request = portalControllerContext.getRequest();
        // Locale
        Locale locale = request.getLocale();
        // Internationalization bundle
        Bundle bundle = this.bundleFactory.getBundle(locale);

        // Calendar event edition form
        InteractikCalendarEventViewForm form = this.applicationContext.getBean(InteractikCalendarEventViewForm.class);

        // Nuxeo document
        Document document = this.repository.getCurrentDocument(portalControllerContext);
        PropertyMap documentProperties = document.getProperties();

        form.setDocument(this.dao.toDTO(document));
        form.setTitle(document.getString(TITLE_PROPERTY));

        // All day indicator
        boolean allDay = this.isAllDay(portalControllerContext, document);
        form.setAllDay(allDay);

        form.setStartDate(document.getDate(START_DATE_PROPERTY));
        form.setEndDate(document.getDate(END_DATE_PROPERTY));

        form.setSameDay(this.isSameDay(form));
        form.setEndDateAllDay(this.getEndDateAllDay(form));

        // Location
        String location = this.getLocation(portalControllerContext, document);
        form.setLocation(location);

        // Calendar color
        CalendarColor calendarColor = this.getCalendarColor(portalControllerContext, document);
        // Color
        InteractikCalendarColor color = (InteractikCalendarColor) this.getColor(portalControllerContext, document, calendarColor);
        form.setColor(color);
        
        // Description
        String description = this.getDescription(portalControllerContext, document);
        form.setDescription(description);

        // Attachments
        this.setAttachments(portalControllerContext, document, form);
        
        // Formatted date & time
        DateFormat titleDateFormat = DateFormat.getDateInstance(DateFormat.LONG, locale);
        Date start = document.getDate("vevent:dtstart");
        Date end = document.getDate("vevent:dtend");
        StringBuilder formattedDate = new StringBuilder();
        StringBuilder formattedTime = new StringBuilder();
        formattedDate.append(titleDateFormat.format(start));
        if (DateUtils.isSameDay(start, end)) {
            if (BooleanUtils.toBoolean(documentProperties.getBoolean("vevent:allDay"))) {
                formattedTime.append(bundle.getString("CALENDAR_EVENT_ALL_DAY"));
            } else {
                Calendar startCalendar = Calendar.getInstance();
                startCalendar.setTime(start);
                int startHour = startCalendar.get(Calendar.HOUR_OF_DAY);
                formattedTime.append(String.valueOf(startHour));
                formattedTime.append("h");
                int startMinute = startCalendar.get(Calendar.MINUTE);
                if (startMinute != 0) {
                    formattedTime.append(StringUtils.leftPad(String.valueOf(startMinute), 2, "0"));
                }

                formattedTime.append(" > ");

                Calendar endCalendar = Calendar.getInstance();
                endCalendar.setTime(end);
                int endHour = endCalendar.get(Calendar.HOUR_OF_DAY);
                formattedTime.append(String.valueOf(endHour));
                formattedTime.append("h");
                int endMinute = endCalendar.get(Calendar.MINUTE);
                if (endMinute != 0) {
                    formattedTime.append(StringUtils.leftPad(String.valueOf(endMinute), 2, "0"));
                }
            }
        } else {
            formattedDate.append(" > ");
            formattedDate.append(titleDateFormat.format(end));
        }
        form.setFormattedDate(formattedDate.toString());
        form.setFormattedTime(formattedTime.toString());

        // Permalink
        String webId = document.getString("ttc:webid");
        String path;
        if (StringUtils.isEmpty(webId)) {
            path = StringUtils.removeEnd(document.getPath(), ".proxy");
        } else {
            path = NuxeoController.webIdToCmsPath(webId);
        }
        String permalink = nuxeoController.createPermalink(path);

        // Twitter URL
        try {
            URL url = new URL("https", "twitter.com", "/intent/tweet");
            URIBuilder builder = new URIBuilder(url.toURI());
            builder.setParameter("url", permalink);
            form.setTwitterUrl(builder.toString());
        } catch (URISyntaxException | MalformedURLException e) {
            throw new PortletException(e);
        }

        // Facebook URL
        try {
            URL url = new URL("https", "www.facebook.com", "/dialog/share");
            URIBuilder builder = new URIBuilder(url.toURI());
            builder.setParameter("app_id", this.facebookAppId);
            builder.setParameter("display", "page");
            builder.setParameter("href", permalink);
            builder.setParameter("redirect_uri", permalink);
            form.setFacebookUrl(builder.toString());
        } catch (URISyntaxException | MalformedURLException e) {
            throw new PortletException(e);
        }

        // Interactik properties
        this.commonRepository.fillInteractikEventProperties(portalControllerContext, document, form.getInteractik());

        return form;
    }


    private InteractikCalendarColor getColor(PortalControllerContext portalControllerContext, Document document, CalendarColor calendarColor) throws PortletException {
        // Color identifier
        String colorId;
        if (document == null) {
            colorId = null;
        } else {
            colorId = document.getString(COLOR_PROPERTY);
            if (colorId == null)
            {
                String dept = document.getString(DEPARTEMENT_PROPERTY);
                if (dept != null)
                {
                	if (CODE_DEPARTEMENT_ACADEMIE.equals(dept)) colorId = InteractikCalendarColor.ACADEMIE_BRETAGNE.toString();
                	else if ("22".equals(dept)) colorId = InteractikCalendarColor.COTE_D_ARMOR.toString();
                	else if ("29".equals(dept)) colorId = InteractikCalendarColor.FINISTERE.toString();
                	else if ("35".equals(dept)) colorId = InteractikCalendarColor.ILE_ET_VILAINE.toString();
                	else if ("53".equals(dept)) colorId = InteractikCalendarColor.MORBIHAN.toString();
                }
            }
        }

        if ((colorId == null) && (calendarColor != null)) {
            colorId = calendarColor.getId();
        }

        return InteractikCalendarColor.fromId(colorId);
    }

}
