package fr.toutatice.cartoun.portlet.detailactivite.service.impl;

import java.security.Principal;
import java.util.Iterator;

import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.lang.StringUtils;
import org.osivia.portal.api.internationalization.Bundle;
import org.osivia.portal.api.internationalization.IBundleFactory;
import org.osivia.portal.api.internationalization.IInternationalizationService;
import org.osivia.portal.api.locator.Locator;
import org.springframework.stereotype.Service;

import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.Element;
import com.itextpdf.text.Font;
import com.itextpdf.text.FontFactory;
import com.itextpdf.text.Image;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import com.itextpdf.tool.xml.ElementList;

import fr.toutatice.cartoun.portlet.detailactivite.bean.ActiviteBean;
import fr.toutatice.cartoun.portlet.detailactivite.bean.LinkBean;
import fr.toutatice.cartoun.portlet.detailactivite.bean.ReferentBean;
import fr.toutatice.cartoun.portlet.detailactivite.service.IExportService;
import fr.toutatice.cartoun.portlet.detailactivite.util.VocabularyUtil;
import fr.toutatice.cartoun.portlet.detailactivite.util.pdf.ActiviteFooter;
import fr.toutatice.cartoun.portlet.detailactivite.util.pdf.PDFUtil;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;

@Service
public class ExportServiceImpl implements IExportService {

    private static final String IMG_TETIERE_PATH = "/WEB-INF/classes/img/cartoun.png";

    /** Bundle factory. */
    protected IBundleFactory bundleFactory;


    public ExportServiceImpl() {
        // Bundle factory
        IInternationalizationService internationalizationService = Locator.findMBean(IInternationalizationService.class,
                IInternationalizationService.MBEAN_NAME);
        bundleFactory = internationalizationService.getBundleFactory(this.getClass().getClassLoader());
    }

    @Override
    public void writePdf(ResourceRequest request, ResourceResponse response, PortletContext portletContext, ActiviteBean activite) throws PortletException {
        Document document = new Document();

        ByteArrayOutputStream baos = null;

        try {
            final Principal principal = request.getUserPrincipal();
            NuxeoController nuxeoController = new NuxeoController(request, response, portletContext);
            Bundle bundle = bundleFactory.getBundle(request.getLocale());

            baos = new ByteArrayOutputStream();
            document.setMargins(document.leftMargin(), document.rightMargin(), document.topMargin(), document.bottomMargin() + 20);
            PdfWriter writer = PdfWriter.getInstance(document, baos);
            document.open();

            Font titleFont = FontFactory.getFont(FontFactory.HELVETICA, 18, Font.NORMAL);
            Font chapterFont = FontFactory.getFont(FontFactory.HELVETICA, 14, Font.NORMAL);
            Font paragraphFont = FontFactory.getFont(FontFactory.HELVETICA, 10, Font.NORMAL);
            Font paragraphFontBold = FontFactory.getFont(FontFactory.HELVETICA, 10, Font.BOLD);

            // footer
            writer.setPageEvent(new ActiviteFooter(paragraphFont, paragraphFontBold, portletContext, activite, bundle));

            // header
            PdfPTable headTable = new PdfPTable(2);
            headTable.setWidths(new float[]{1, 2});
            Image imgTetiere = Image.getInstance(IOUtils.toByteArray(portletContext.getResourceAsStream(IMG_TETIERE_PATH)));
            imgTetiere.scalePercent(80);
            PdfPCell cell = new PdfPCell(imgTetiere);
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setColspan(2);
            cell.setHorizontalAlignment(Element.ALIGN_CENTER);
            // cell.setPaddingBottom(20);
            headTable.addCell(cell);

            String basePortalUrl = nuxeoController.getPortalUrlFactory().getBasePortalUrl(nuxeoController.getPortalCtx());
            cell = new PdfPCell(new Phrase(basePortalUrl, paragraphFont));
            cell.setBorder(Rectangle.BOTTOM);
            cell.setColspan(2);
            cell.setHorizontalAlignment(Element.ALIGN_CENTER);
            cell.setPaddingBottom(10);
            headTable.addCell(cell);


            // vignette
            Image vignette = Image.getInstance(PDFUtil.getImage(activite.getUrlVignette(), nuxeoController));
            vignette.scaleToFit(130, 70);
            cell = new PdfPCell(vignette);
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_LEFT);
            cell.setRightIndent(20);
            cell.setPaddingTop(10);
            headTable.addCell(cell);

            // titre
            cell = new PdfPCell(new Phrase(activite.getTitle(), titleFont));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
            headTable.addCell(cell);

            document.add(headTable);
            document.add(Chunk.NEWLINE);

            // identifiant
            Paragraph identifiant = new Paragraph(bundle.getString("label.identifiant"), chapterFont);
            identifiant.add(Chunk.NEWLINE);
            identifiant.add(Chunk.TABBING);
            identifiant.add(new Chunk(activite.getWebId(), paragraphFont));
            identifiant.add(Chunk.NEWLINE);
            identifiant.add(Chunk.TABBING);
            if (activite.isShareable()) {
                identifiant.add(new Chunk(bundle.getString("activite.partageable"), paragraphFont));
            } else {
                identifiant.add(new Chunk(bundle.getString("activite.non.partageable"), paragraphFont));
            }
            document.add(identifiant);
            document.add(Chunk.NEWLINE);


            // contacts
            if (principal != null) {
                Paragraph contacts = new Paragraph(bundle.getString("label.contact"), chapterFont);
                if ((activite.getLstReferent1() != null) && !activite.getLstReferent1().isEmpty()) {
                    contacts.add(Chunk.NEWLINE);
                    contacts.add(new Phrase(bundle.getString("label.contact.niveau.1") + " : ", paragraphFontBold));
                    for (ReferentBean referent : activite.getLstReferent1()) {
                        contacts.add(Chunk.NEWLINE);
                        contacts.add(Chunk.TABBING);
                        contacts.add(new Chunk(referent.getIdentite(), paragraphFont));
                    }
                }
                if (activite.getLstReferent2() != null) {
                    contacts.add(Chunk.NEWLINE);
                    contacts.add(new Phrase(bundle.getString("label.contact.niveau.2") + " : ", paragraphFontBold));
                    for (ReferentBean referent : activite.getLstReferent2()) {
                        contacts.add(Chunk.NEWLINE);
                        contacts.add(Chunk.TABBING);
                        contacts.add(new Chunk(referent.getIdentite(), paragraphFont));
                    }
                }
                if (activite.getLstReferent3() != null) {
                    contacts.add(Chunk.NEWLINE);
                    contacts.add(new Phrase(bundle.getString("label.contact.niveau.3") + " : ", paragraphFontBold));
                    for (ReferentBean referent : activite.getLstReferent3()) {
                        contacts.add(Chunk.NEWLINE);
                        contacts.add(Chunk.TABBING);
                        contacts.add(new Chunk(referent.getIdentite(), paragraphFont));
                    }
                }
                document.add(contacts);
                document.add(Chunk.NEWLINE);
            }

            // viaeduc
            if (StringUtils.isNotBlank(activite.getViaeduc())) {
                Paragraph viaeduc = new Paragraph(bundle.getString("label.viaeduc"), chapterFont);
                viaeduc.add(Chunk.TABBING);
                viaeduc.add(new Chunk(activite.getViaeduc(), paragraphFont));
                document.add(viaeduc);
                document.add(Chunk.NEWLINE);
            }

            // experitheque
            if (StringUtils.isNotBlank(activite.getUrlExpe())) {
                Paragraph expe = new Paragraph(bundle.getString("label.experitheque"), chapterFont);
                expe.add(Chunk.TABBING);
                expe.add(new Chunk(activite.getUrlExpe(), paragraphFont));
                document.add(expe);
                document.add(Chunk.NEWLINE);
            }

            // repères
            Phrase reperes = new Phrase(bundle.getString("label.reperes"), chapterFont);
            document.add(reperes);
            PdfPTable reperesTable = new PdfPTable(2);
            cell = new PdfPCell(new Phrase(bundle.getString("label.reperes.principal"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            reperesTable.addCell(cell);
            cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "carto_reperes", activite.getReperePrincipal()), paragraphFont));
            cell.setBorder(Rectangle.NO_BORDER);
            reperesTable.addCell(cell);
            if (StringUtils.isNotBlank(activite.getRepereOther())) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.reperes.other"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                reperesTable.addCell(cell);
                cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "carto_reperes", activite.getRepereOther()), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                reperesTable.addCell(cell);
            }
            document.add(reperesTable);
            document.add(Chunk.NEWLINE);


            // parcoursAxes
            Phrase parcours = new Phrase(bundle.getString("label.titre.parcoursAxes"), chapterFont);
            document.add(parcours);
            PdfPTable parcoursTable = new PdfPTable(2);
            cell = new PdfPCell(new Phrase(bundle.getString("label.parcours"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            parcoursTable.addCell(cell);
            if ((activite.getLstParcours() == null) || (activite.getLstParcours().size() == 0)) {
                cell = new PdfPCell(new Phrase(bundle.getString("value.parcours.null"), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                parcoursTable.addCell(cell);
            } else {
                for (Iterator<String> parcoursIterator = activite.getLstParcours().iterator(); parcoursIterator.hasNext();) {
                    String parcour = parcoursIterator.next();
                    cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "carto_parcours", parcour), paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    parcoursTable.addCell(cell);
                    if (parcoursIterator.hasNext()) {
                        parcoursTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }

            // axes
            cell = new PdfPCell(new Phrase(bundle.getString("label.axes"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            parcoursTable.addCell(cell);
            if ((activite.getLstAxes() == null) || (activite.getLstAxes().size() == 0)) {
                cell = new PdfPCell(new Phrase(bundle.getString("value.axes.null"), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                parcoursTable.addCell(cell);
            } else {
                for (Iterator<String> axesIterator = activite.getLstAxes().iterator(); axesIterator.hasNext();) {
                    String axe = axesIterator.next();
                    cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "carto_axes", axe), paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    parcoursTable.addCell(cell);
                    if (axesIterator.hasNext()) {
                        parcoursTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }
            document.add(parcoursTable);
            document.add(Chunk.NEWLINE);

            // discipline
            Phrase disciplines = new Phrase(bundle.getString("label.discipline.public"), chapterFont);
            document.add(disciplines);
            PdfPTable disciplineTable = new PdfPTable(2);
            cell = new PdfPCell(new Phrase(bundle.getString("label.discipline"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            disciplineTable.addCell(cell);
            for (Iterator<String> disciplinesIterator = activite.getLstDisciplines().iterator(); disciplinesIterator.hasNext();) {
                String discipline = disciplinesIterator.next();
                cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "disciplines", discipline), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                disciplineTable.addCell(cell);
                if (disciplinesIterator.hasNext()) {
                    disciplineTable.addCell(PDFUtil.emptyCell());
                }
            }
            cell = new PdfPCell(new Phrase(bundle.getString("label.public"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            disciplineTable.addCell(cell);
            for (Iterator<String> publicsIterator = activite.getLstPublics().iterator(); publicsIterator.hasNext();) {
                String publicS = publicsIterator.next();
                cell = new PdfPCell(new Phrase(publicS, paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                disciplineTable.addCell(cell);
                if (publicsIterator.hasNext()) {
                    disciplineTable.addCell(PDFUtil.emptyCell());
                }
            }
            document.add(disciplineTable);
            document.add(Chunk.NEWLINE);

            // description
            Phrase description = new Phrase(bundle.getString("label.description"), chapterFont);
            document.add(description);
            PdfPTable descriptionTable = new PdfPTable(2);

            // niveau
            cell = new PdfPCell(new Phrase(bundle.getString("label.niveau"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            if (StringUtils.isNotBlank(activite.getLevel())) {
                cell = new PdfPCell(new Phrase(activite.getLevel(), paragraphFont));
            } else {
                cell = new PdfPCell(new Phrase(bundle.getString("value.niveau.null"), paragraphFont));
            }
            cell.setBorder(Rectangle.NO_BORDER);
            descriptionTable.addCell(cell);

            // pratique
            cell = new PdfPCell(new Phrase(bundle.getString("label.pratique"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            if ((activite.getLstPractices() == null) || (activite.getLstPractices().size() == 0)) {
                cell = new PdfPCell(new Phrase(bundle.getString("value.pratique.null"), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
            } else {
                for (Iterator<String> practicesIterator = activite.getLstPractices().iterator(); practicesIterator.hasNext();) {
                    String practice = practicesIterator.next();
                    cell = new PdfPCell(new Phrase(practice, paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (practicesIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }

            // outils
            cell = new PdfPCell(new Phrase(bundle.getString("label.outils"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            if ((activite.getLstTools() == null) || (activite.getLstTools().size() == 0)) {
                cell = new PdfPCell(new Phrase(bundle.getString("value.outils.null"), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
            } else {
                for (Iterator<String> toolsIterator = activite.getLstTools().iterator(); toolsIterator.hasNext();) {
                    String tool = toolsIterator.next();
                    cell = new PdfPCell(new Phrase(VocabularyUtil.getVocabularyEntry(nuxeoController, "unu_outils", tool), paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (toolsIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }

            // duree
            if (StringUtils.isNotBlank(activite.getDuration())) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.duree"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                descriptionTable.addCell(cell);
                cell = new PdfPCell(new Phrase(activite.getDuration(), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
            }

            // nature
            cell = new PdfPCell(new Phrase(bundle.getString("label.nature"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            if (CollectionUtils.isNotEmpty(activite.getLstPedagogicals())) {
                for (Iterator<String> pdgIterator = activite.getLstPedagogicals().iterator(); pdgIterator.hasNext();) {
                    String tool = pdgIterator.next();
                    cell = new PdfPCell(new Phrase(tool, paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (pdgIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            } else {
                cell = new PdfPCell(new Phrase(null, paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
            }

            // modalite
            cell = new PdfPCell(new Phrase(bundle.getString("label.modalite"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            for (Iterator<String> cadresIterator = activite.getLstCadres().iterator(); cadresIterator.hasNext();) {
                String cadre = cadresIterator.next();
                cell = new PdfPCell(new Phrase(cadre, paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
                if (cadresIterator.hasNext()) {
                    descriptionTable.addCell(PDFUtil.emptyCell());
                }
            }

            // lieu
            cell = new PdfPCell(new Phrase(bundle.getString("label.lieu"), paragraphFontBold));
            cell.setBorder(Rectangle.NO_BORDER);
            cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
            cell.setRightIndent(20);
            descriptionTable.addCell(cell);
            for (Iterator<String> placesIterator = activite.getLstPlaces().iterator(); placesIterator.hasNext();) {
                String place = placesIterator.next();
                cell = new PdfPCell(new Phrase(place, paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
                if (placesIterator.hasNext()) {
                    descriptionTable.addCell(PDFUtil.emptyCell());
                }
            }

            // logiciels
            if ((activite.getLstSoftware() != null) && !activite.getLstSoftware().isEmpty()) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.logiciel"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                descriptionTable.addCell(cell);
                for (Iterator<String> softwaresIterator = activite.getLstSoftware().iterator(); softwaresIterator.hasNext();) {
                    String software = softwaresIterator.next();
                    cell = new PdfPCell(new Phrase(software, paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (softwaresIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }

            // elogiciels
            if ((activite.getLstESoftware() != null) && !activite.getLstESoftware().isEmpty()) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.elogiciel"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                descriptionTable.addCell(cell);
                for (Iterator<LinkBean> esoftwaresIterator = activite.getLstESoftware().iterator(); esoftwaresIterator.hasNext();) {
                    LinkBean esoftware = esoftwaresIterator.next();
                    cell = new PdfPCell(new Phrase(esoftware.getName(), paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (esoftwaresIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }

            // transfert
            if (StringUtils.isNotBlank(activite.getTransferability())) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.transfert"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                descriptionTable.addCell(cell);
                cell = new PdfPCell(new Phrase(activite.getTransferability(), paragraphFont));
                cell.setBorder(Rectangle.NO_BORDER);
                descriptionTable.addCell(cell);
            }

            // competence
            if ((activite.getLstCompetence() != null) && !activite.getLstCompetence().isEmpty()) {
                cell = new PdfPCell(new Phrase(bundle.getString("label.competence"), paragraphFontBold));
                cell.setBorder(Rectangle.NO_BORDER);
                cell.setHorizontalAlignment(Element.ALIGN_RIGHT);
                cell.setRightIndent(20);
                descriptionTable.addCell(cell);
                for (Iterator<String> competenceIterator = activite.getLstCompetence().iterator(); competenceIterator.hasNext();) {
                    String competence = competenceIterator.next();
                    cell = new PdfPCell(new Phrase(competence, paragraphFont));
                    cell.setBorder(Rectangle.NO_BORDER);
                    descriptionTable.addCell(cell);
                    if (competenceIterator.hasNext()) {
                        descriptionTable.addCell(PDFUtil.emptyCell());
                    }
                }
            }
            document.add(descriptionTable);
            document.add(Chunk.NEWLINE);

            // resume
            Phrase resume = new Phrase(bundle.getString("label.resume"), chapterFont);
            document.add(resume);
            if (StringUtils.isNotBlank(activite.getSummary())) {
                Paragraph resumeContent = new Paragraph(activite.getSummary(), paragraphFont);
                document.add(resumeContent);
            }
            document.add(Chunk.NEWLINE);

            // scenario
            Phrase scenario = new Phrase(bundle.getString("label.scenario"), chapterFont);
            document.add(scenario);
            if (StringUtils.isNotBlank(activite.getScenario())) {
                String scenarioS = PDFUtil.sanitizeHTML(activite.getScenario());
                ElementList list = PDFUtil.parseStringHTMLToElements(scenarioS, nuxeoController);
                for (Element element : list) {
                    document.add(element);
                }
            }
            document.add(Chunk.NEWLINE);

            // analyse
            Phrase analyse = new Phrase(bundle.getString("label.analyse"), chapterFont);
            document.add(analyse);
            if (StringUtils.isNotBlank(activite.getAnalysis())) {
                String analysis = PDFUtil.sanitizeHTML(activite.getAnalysis());
                ElementList elementList = PDFUtil.parseStringHTMLToElements(analysis, nuxeoController);
                for (Element element : elementList) {
                    document.add(element);
                }
            }
            document.add(Chunk.NEWLINE);

            // pieges
            Phrase pieges = new Phrase(bundle.getString("label.pieges"), chapterFont);
            document.add(pieges);
            if (StringUtils.isNotBlank(activite.getTrap())) {
                String trap = PDFUtil.sanitizeHTML(activite.getTrap());
                ElementList list = PDFUtil.parseStringHTMLToElements(trap, nuxeoController);
                for (Element element : list) {
                    document.add(element);
                }
            }
            document.add(Chunk.NEWLINE);

            document.addTitle(activite.getTitle());
            document.close();

            response.setContentType("application/pdf");
            response.setProperty("Content-disposition", "inline; filename=\"" + activite.getTitle() + ".pdf" + "\"");
            response.setContentLength(baos.size());
            baos.writeTo(response.getPortletOutputStream());
            baos.flush();
            baos.close();

        } catch (Exception e) {
            throw new PortletException(e);
        }
    }

}
