package fr.toutatice.cartoun.generator.repository;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.osivia.portal.api.cache.services.CacheInfo;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;
import org.osivia.portal.core.constants.InternalConstants;
import org.springframework.stereotype.Repository;

import fr.toutatice.cartoun.generator.model.Academy;
import fr.toutatice.cartoun.generator.model.Configuration;
import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.services.NuxeoCommandContext;

/**
 * Generator repository implementation.
 *
 * @author Cédric Krommenhoek
 * @see GeneratorRepository
 */
@Repository
public class GeneratorRepositoryImpl implements GeneratorRepository {

    /** Generator properties file name. */
    private static final String PROPERTIES_FILE_NAME = "generator.properties";

    /** Academies property name. */
    private static final String ACADEMIES_PROPERTY = "generator.academies";
    /** Pools count property name. */
    private static final String POOLS_COUNT_PROPERTY = "generator.pools.count";
    /** Institutions count property name. */
    private static final String INSTITUTIONS_COUNT_PROPERTY = "generator.institutions.count";
    /** Profiles base DN property name. */
    private static final String PROFILES_BASE_DN_PROPERTY = "generator.profiles.baseDn";
    /** Profiles displines property name. */
    private static final String PROFILES_DISCIPLINES = "generator.profiles.disciplines";
    /** Profiles values property name. */
    private static final String PROFILES_VALUES = "generator.profiles.values";
    /** Users base DN property name. */
    private static final String USERS_BASE_DN_PROPERTY = "generator.users.baseDn";
    /** Users count property name. */
    private static final String USERS_COUNT_PROPERTY = "generator.users.count";
    /** Folders count property name. */
    private static final String FOLDERS_COUNT_PROPERTY = "generator.folders.count";
    /** Activities count property name. */
    private static final String ACTIVITIES_COUNT_PROPERTY = "generator.activities.count";
    /** Referrers count property name. */
    private static final String REFERRERS_COUNT_PROPERTY = "generator.referrers.count";


    /** Generator properties. */
    private final Properties properties;
    /** LDAP context. */
    private final InitialLdapContext ldapContext;


    /**
     * Constructor.
     *
     * @throws IOException
     * @throws NamingException
     */
    public GeneratorRepositoryImpl() throws IOException, NamingException {
        super();

        // Generator properties
        this.properties = new Properties();
        InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(PROPERTIES_FILE_NAME);
        if (inputStream != null) {
            this.properties.load(inputStream);
        } else {
            throw new FileNotFoundException(PROPERTIES_FILE_NAME);
        }

        // LDAP context
        this.properties.put(Context.PROVIDER_URL,
                "ldap://" + System.getProperty(InternalConstants.ENV_LDAP_HOST) + ":" + System.getProperty(InternalConstants.ENV_LDAP_PORT));
        this.properties.put(Context.SECURITY_PRINCIPAL, System.getProperty(InternalConstants.ENV_LDAP_PRINCIPAL));
        this.properties.put(Context.SECURITY_CREDENTIALS, System.getProperty(InternalConstants.ENV_LDAP_CREDENTIALS));
        this.ldapContext = new InitialLdapContext(this.properties, null);
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Configuration getConfiguration(PortalControllerContext portalControllerContext) throws PortletException {
        // Window
        PortalWindow window = WindowFactory.getWindow(portalControllerContext.getRequest());

        // Properties
        String[] academyLabels = StringUtils.split(this.properties.getProperty(ACADEMIES_PROPERTY), "|");
        List<Academy> academies = new ArrayList<Academy>(academyLabels.length);
        for (int i = 0; i < academyLabels.length; i++) {
            Academy academy = new Academy(i + 1, academyLabels[i]);
            academies.add(academy);
        }
        int poolsCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(POOLS_COUNT_PROPERTY),
                this.properties.getProperty(POOLS_COUNT_PROPERTY)));
        int institutionsCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(INSTITUTIONS_COUNT_PROPERTY),
                this.properties.getProperty(INSTITUTIONS_COUNT_PROPERTY)));
        String profilesBaseDn = StringUtils.defaultIfEmpty(window.getProperty(PROFILES_BASE_DN_PROPERTY),
                this.properties.getProperty(PROFILES_BASE_DN_PROPERTY));
        List<String> disciplines = Arrays.asList(StringUtils.split(this.properties.getProperty(PROFILES_DISCIPLINES), "|"));
        List<String> profiles = Arrays.asList(StringUtils.split(this.properties.getProperty(PROFILES_VALUES), "|"));
        String usersBaseDn = StringUtils.defaultIfEmpty(window.getProperty(USERS_BASE_DN_PROPERTY), this.properties.getProperty(USERS_BASE_DN_PROPERTY));
        int usersCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(USERS_COUNT_PROPERTY),
                this.properties.getProperty(USERS_COUNT_PROPERTY)));
        int foldersCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(FOLDERS_COUNT_PROPERTY),
                this.properties.getProperty(FOLDERS_COUNT_PROPERTY)));
        int activitiesCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(ACTIVITIES_COUNT_PROPERTY),
                this.properties.getProperty(ACTIVITIES_COUNT_PROPERTY)));
        int referrersCount = NumberUtils.toInt(StringUtils.defaultIfEmpty(window.getProperty(REFERRERS_COUNT_PROPERTY),
                this.properties.getProperty(REFERRERS_COUNT_PROPERTY)));

        // Configuration
        Configuration configuration = new Configuration();
        configuration.setAcademies(academies);
        configuration.setPoolsCount(poolsCount);
        configuration.setInstitutionsCount(institutionsCount);
        configuration.setProfilesBaseDn(profilesBaseDn);
        configuration.setDisciplines(disciplines);
        configuration.setProfiles(profiles);
        configuration.setUsersBaseDn(usersBaseDn);
        configuration.setUsersCount(usersCount);
        configuration.setFoldersCount(foldersCount);
        configuration.setActivitiesCount(activitiesCount);
        configuration.setReferrersCount(referrersCount);

        return configuration;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void setConfiguration(PortalControllerContext portalControllerContext, Configuration configuration) throws PortletException {
        // Configuration
        if (configuration.getPoolsCount() == null) {
            configuration.setPoolsCount(NumberUtils.toInt(this.properties.getProperty(POOLS_COUNT_PROPERTY)));
        }
        if (configuration.getInstitutionsCount() == null) {
            configuration.setInstitutionsCount(NumberUtils.toInt(this.properties.getProperty(INSTITUTIONS_COUNT_PROPERTY)));
        }
        if (StringUtils.isBlank(configuration.getProfilesBaseDn())) {
            configuration.setProfilesBaseDn(this.properties.getProperty(PROFILES_BASE_DN_PROPERTY));
        }
        if (StringUtils.isBlank(configuration.getUsersBaseDn())) {
            configuration.setUsersBaseDn(this.properties.getProperty(USERS_BASE_DN_PROPERTY));
        }
        if (configuration.getUsersCount() == null) {
            configuration.setUsersCount(NumberUtils.toInt(this.properties.getProperty(USERS_COUNT_PROPERTY)));
        }
        if (configuration.getFoldersCount() == null) {
            configuration.setFoldersCount(NumberUtils.toInt(this.properties.getProperty(FOLDERS_COUNT_PROPERTY)));
        }
        if (configuration.getActivitiesCount() == null) {
            configuration.setActivitiesCount(NumberUtils.toInt(this.properties.getProperty(ACTIVITIES_COUNT_PROPERTY)));
        }
        if (configuration.getReferrersCount() == null) {
            configuration.setReferrersCount(NumberUtils.toInt(this.properties.getProperty(REFERRERS_COUNT_PROPERTY)));
        }

        // Window
        PortalWindow window = WindowFactory.getWindow(portalControllerContext.getRequest());
        window.setProperty(POOLS_COUNT_PROPERTY, String.valueOf(configuration.getPoolsCount()));
        window.setProperty(INSTITUTIONS_COUNT_PROPERTY, String.valueOf(configuration.getInstitutionsCount()));
        window.setProperty(PROFILES_BASE_DN_PROPERTY, configuration.getProfilesBaseDn());
        window.setProperty(USERS_BASE_DN_PROPERTY, configuration.getUsersBaseDn());
        window.setProperty(USERS_COUNT_PROPERTY, String.valueOf(configuration.getUsersCount()));
        window.setProperty(FOLDERS_COUNT_PROPERTY, String.valueOf(configuration.getFoldersCount()));
        window.setProperty(ACTIVITIES_COUNT_PROPERTY, String.valueOf(configuration.getActivitiesCount()));
        window.setProperty(REFERRERS_COUNT_PROPERTY, String.valueOf(configuration.getReferrersCount()));
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void generate(PortalControllerContext portalControllerContext) throws PortletException {
        // Configuration
        Configuration configuration = this.getConfiguration(portalControllerContext);

        // Nuxeo controller
        NuxeoController nuxeoController = this.getNuxeoController(portalControllerContext);
        nuxeoController.setAuthType(NuxeoCommandContext.AUTH_TYPE_SUPERUSER);
        nuxeoController.setCacheType(CacheInfo.CACHE_SCOPE_PORTLET_CONTEXT);
        nuxeoController.setAsynchronousCommand(true);

        Locale locale = nuxeoController.getRequest().getLocale();

        // Commands
        for (Academy academy : configuration.getAcademies()) {
            INuxeoCommand command = new GenerateCommand(this.ldapContext, configuration, academy, locale);
            nuxeoController.executeNuxeoCommand(command);
        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void purge(PortalControllerContext portalControllerContext) throws PortletException {
        // Configuration
        Configuration configuration = this.getConfiguration(portalControllerContext);

        // Purge LDAP groups
        purgeProfiles(configuration);

        // Purge LDAP users
        purgeUsers(configuration);
    }


    /**
     * Purge profiles.
     * 
     * @param configuration generator configuration
     * @throws PortletException
     */
    private void purgeProfiles(Configuration configuration) throws PortletException {
        try {
            String filter = "cn=*";
            SearchControls controls = new SearchControls();
            controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
            NamingEnumeration<SearchResult> searchResults = this.ldapContext.search(configuration.getProfilesBaseDn(), filter, controls);
            while (searchResults.hasMore()) {
                SearchResult searchResult = searchResults.next();
                if (searchResult.getName().matches("cn=[0-9]{2}-.+")) {
                    this.ldapContext.unbind(searchResult.getNameInNamespace());
                }
            }
        } catch (NamingException e) {
            throw new PortletException(e);
        }
    }


    /**
     * Purge users.
     * 
     * @param configuration generator configuration
     * @throws PortletException
     */
    private void purgeUsers(Configuration configuration) throws PortletException {
        try {
            String filter = "uid=*";
            SearchControls controls = new SearchControls();
            controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
            NamingEnumeration<SearchResult> searchResults = this.ldapContext.search(configuration.getUsersBaseDn(), filter, controls);
            while (searchResults.hasMore()) {
                SearchResult searchResult = searchResults.next();
                if (searchResult.getName().matches("uid=.+@[0-9]{2}")) {
                    this.ldapContext.unbind(searchResult.getNameInNamespace());
                }
            }
        } catch (NamingException e) {
            throw new PortletException(e);
        }
    }


    /**
     * Get Nuxeo controller
     *
     * @param portalControllerContext portal controller context
     * @return Nuxeo controller
     */
    private NuxeoController getNuxeoController(PortalControllerContext portalControllerContext) {
        PortletRequest request = portalControllerContext.getRequest();
        PortletResponse response = portalControllerContext.getResponse();
        PortletContext portletContext = portalControllerContext.getPortletCtx();
        return new NuxeoController(request, response, portletContext);
    }

}
