package fr.toutatice.services.calendar.event.edition.portlet.service;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.AGENDA_INTERACTIK;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DATE_DEBUT_INSCRIPTION_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DATE_FIN_INSCRIPTION_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DEPARTEMENT_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.IMPORTED_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.URL_INSCRIPTION_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.VILLE_PROPERTY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_VOCABULARY;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.CODE_DEPARTEMENT_ACADEMIE;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.portlet.PortletException;
import javax.portlet.PortletRequest;

import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.automation.client.model.Document;
import org.osivia.portal.api.Constants;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;
import org.osivia.services.calendar.common.model.Attachments;
import org.osivia.services.calendar.common.model.CalendarColor;
import org.osivia.services.calendar.common.model.CalendarCommonEventForm;
import org.osivia.services.calendar.common.model.CalendarEditionOptions;
import org.osivia.services.calendar.common.model.CalendarEventDates;
import org.osivia.services.calendar.common.model.converter.CalendarColorPropertyEditor;
import org.osivia.services.calendar.common.service.CalendarServiceImpl;
import org.osivia.services.calendar.event.edition.portlet.repository.CalendarEventEditionRepository;
import org.osivia.services.calendar.event.edition.portlet.service.CalendarEventEditionService;
import org.osivia.services.calendar.event.edition.portlet.service.CalendarEventEditionServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.VocabularyEntry;
import fr.toutatice.portail.cms.nuxeo.api.VocabularyHelper;
import fr.toutatice.portail.cms.nuxeo.api.cms.NuxeoDocumentContext;
import fr.toutatice.portail.cms.nuxeo.api.services.dao.DocumentDAO;
import fr.toutatice.services.calendar.common.model.InteractikCalendarColor;
import fr.toutatice.services.calendar.common.model.converter.InteractikCalendarColorPropertyEditor;
import fr.toutatice.services.calendar.event.edition.portlet.model.InteractikCalendarEventEditionForm;

/**
 * Calendar event edition portlet service implementation.
 * 
 * @author Cédric Krommenhoek
 * @see CalendarServiceImpl
 * @see CalendarEventEditionService
 */
@Service
@Primary
public class InteractikCalendarEventEditionServiceImpl extends CalendarEventEditionServiceImpl {
	
    /** Portlet repository. */
    @Autowired
    private CalendarEventEditionRepository repository;

    /** Document DAO. */
    @Autowired
    private DocumentDAO dao;
    
    /** Calendar color property editor. */
    @Autowired
    private InteractikCalendarColorPropertyEditor calendarColorPropertyEditor;

    /**
     * Constructor.
     */
    public InteractikCalendarEventEditionServiceImpl() {
        super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public CalendarCommonEventForm getForm(PortalControllerContext portalControllerContext) throws PortletException {
        // Calendar edition options
        CalendarEditionOptions options = this.getEditionOptions(portalControllerContext);
        // Calendar event edition form
        InteractikCalendarEventEditionForm form = this.applicationContext.getBean(InteractikCalendarEventEditionForm.class);
        
        // Event Nuxeo document
        Document document = options.getDocument();
        if (document != null) 
        {
        	form.setDocument(this.dao.toDTO(document));
        } else
        {
        	form.setDocument(null);
        }
        
        // Calendar color
        CalendarColor calendarColor = getCalendarColor(portalControllerContext, options);
        form.setCalendarColor(calendarColor);

        // Title
        String title = this.repository.getTitle(portalControllerContext, document);
        form.setTitle(title);

        // All day indicator
        boolean allDay = this.repository.isAllDay(portalControllerContext, document);
        form.setAllDay(allDay);

        // Dates
        CalendarEventDates dates = this.repository.getDates(portalControllerContext, document, allDay);
        form.setDates(dates);
        this.updateFormDates(form);

        form.setSameDay(this.isSameDay(form));
        form.setEndDateAllDay(this.getEndDateAllDay(form));
        
        // Location
        String location = this.repository.getLocation(portalControllerContext, document);
        form.setLocation(location);

        // Color
        InteractikCalendarColor color = getColor(portalControllerContext, document, calendarColor);
        form.setColor(color);

        // Description
        String description = this.repository.getDescription(portalControllerContext, document);
        form.setDescription(description);

        // Attachments
        Attachments attachments = this.repository.getAttachments(portalControllerContext, document);
	        form.setAttachments(attachments);
        
	     // Portlet request
        PortletRequest request = portalControllerContext.getRequest();
        // Window
        PortalWindow window = WindowFactory.getWindow(request);
        
        String parentPath = window.getProperty(Constants.WINDOW_PROP_URI);
        form.setInteractikAgenda(isInteractikAgenda(parentPath, portalControllerContext));
         
        if (form.isInteractikAgenda())
        {
        	// Nuxeo controller
            NuxeoController nuxeoController = new NuxeoController(portalControllerContext);
            List<String> vocabs = new ArrayList<String>();
            vocabs.add(COOPERATIVE_VOCABULARY);
            VocabularyEntry vocabEntry = VocabularyHelper.getVocabularyEntry(nuxeoController, vocabs, true);
            Map<String, VocabularyEntry> children = vocabEntry.getChildren();
            if (children != null)
            {
	            Collection<VocabularyEntry> childrenValues = children.values();
	            SortedMap<String, String> mapCooperative = new TreeMap<String, String>();
	            for (VocabularyEntry entry : childrenValues)
	            {
	            	mapCooperative.put(entry.getId(), entry.getLabel());
	            }
	            
	        	form.setMapCooperative(mapCooperative);
            }
        }
        // Interactik data
        if (document != null)
        {
        	form.setDepartement(document.getString(DEPARTEMENT_PROPERTY));
        	setDateInscription(form, document);
        	form.setVille(document.getString(VILLE_PROPERTY));
        	form.setUrlInscription(document.getString(URL_INSCRIPTION_PROPERTY));
       		
       		if (StringUtils.isNotEmpty(document.getString(IMPORTED_PROPERTY)))
        	{
        		form.setImported(new Boolean(document.getString(IMPORTED_PROPERTY)));
        	} else
        	{
        		form.setImported(false);
        	}
        }

        return form;
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public CalendarColorPropertyEditor getCalendarColorPropertyEditor()
    {
    	return this.calendarColorPropertyEditor;
    }
    
    private void setDateInscription(InteractikCalendarEventEditionForm form, Document document)
    {
        // Start date
        Date startDate;
        // End date
        Date endDate;
        
        if (document == null) {
            startDate = null;
            endDate = null;
        } else {
            startDate = document.getDate(DATE_DEBUT_INSCRIPTION_PROPERTY);
            endDate = document.getDate(DATE_FIN_INSCRIPTION_PROPERTY);
        }
        
        if (startDate != null)
        {
        	form.setStringDebutInscription(this.dateFormat.format(startDate));
        } else
        {
        	form.setStringDebutInscription(null);
        }
        if (endDate != null)
        {
        	form.setStringFinInscription(this.dateFormat.format(endDate));
        } else
        {
        	form.setStringFinInscription(null);
        }
        form.setDateDebutInscription(startDate);
        form.setDateFinInscription(endDate);
    }
    
    /**
     * {@inheritDoc}
     */
    protected CalendarColor getCalendarColor(PortalControllerContext portalControllerContext, CalendarEditionOptions options) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);

        // Calendar path
        String path;
        if (options.isCreation()) {
            path = options.getParentPath();
        } else {
            // Event path
            String eventPath = options.getDocument().getPath();

            path = StringUtils.substringBeforeLast(eventPath, "/");
        }

        // Calendar Nuxeo document
        NuxeoDocumentContext documentContext = nuxeoController.getDocumentContext(path);
        Document calendar = documentContext.getDocument();

        return this.getCalendarColor(portalControllerContext, calendar);
    }
    
    /**
     * To know if event is in an interactik Agenda
     * @param document
     * @param portalControllerContext
     * @return
     * @throws PortletException
     */
	private boolean isInteractikAgenda(String parentPath, PortalControllerContext portalControllerContext) throws PortletException {
		// Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);
        
        // Nuxeo document context
        NuxeoDocumentContext documentContext = nuxeoController.getDocumentContext(NuxeoController.getParentPath(parentPath));
        
        Document document = nuxeoController.getDocumentContext(parentPath).getDocument();
        
        Document documentParent = documentContext.getDocument();
        
        //We can access to this method from plus button in agenda view or from edit button in event view
        //So we have to test the document or the parent document
		return (AGENDA_INTERACTIK.equals(document.getType()) || AGENDA_INTERACTIK.equals(documentParent.getType()));
	}
    
    /**
     * 
     * @param portalControllerContext
     * @param document
     * @param calendarColor
     * @return
     * @throws PortletException
     */
    private InteractikCalendarColor getColor(PortalControllerContext portalControllerContext, Document document, CalendarColor calendarColor) throws PortletException {
        // Color identifier
        String colorId;
        if (document == null) {
            colorId = null;
        } else {
            colorId = document.getString(COLOR_PROPERTY);
            if (colorId == null)
            {
                String dept = document.getString(DEPARTEMENT_PROPERTY);
                if (dept != null)
                {
                	if (CODE_DEPARTEMENT_ACADEMIE.equals(dept)) colorId = InteractikCalendarColor.ACADEMIE_BRETAGNE.toString();
                	else if ("22".equals(dept)) colorId = InteractikCalendarColor.COTE_D_ARMOR.toString();
                	else if ("29".equals(dept)) colorId = InteractikCalendarColor.FINISTERE.toString();
                	else if ("35".equals(dept)) colorId = InteractikCalendarColor.ILE_ET_VILAINE.toString();
                	else if ("53".equals(dept)) colorId = InteractikCalendarColor.MORBIHAN.toString();
                }
            }
        }

        if ((colorId == null) && (calendarColor != null)) {
            colorId =  calendarColor.getId();
        }

        return InteractikCalendarColor.fromId(colorId);
    }
    
    private Date getEndDateAllDay(InteractikCalendarEventEditionForm form) {
        if (form.getEndDate() != null) {
            Calendar cal = Calendar.getInstance();
            cal.setTime(form.getEndDate());
            cal.add(Calendar.DAY_OF_MONTH, -1);
            return cal.getTime();
        }

        return null;
    }


    private boolean isSameDay(InteractikCalendarEventEditionForm form) {
        boolean sameDay = false;
        Calendar calStart = Calendar.getInstance();
        Calendar calEnd = Calendar.getInstance();
        
        calStart.setTime(form.getStartDate());
        calEnd.setTime(form.getEndDate());
        
        if (form.isAllDay())
        {
            long diff = 0;
            diff = Math.round((calEnd.getTime().getTime() - calStart.getTime().getTime())/86400000);
            sameDay = diff <= 1;
        } else
        {
            calStart.set(Calendar.HOUR_OF_DAY, 0);
            calStart.set(Calendar.MINUTE, 0);
            calStart.set(Calendar.SECOND,0);
            calStart.set(Calendar.MILLISECOND, 0);
            
            calEnd.set(Calendar.HOUR_OF_DAY, 0);
            calEnd.set(Calendar.MINUTE, 0);
            calEnd.set(Calendar.SECOND,0);
            calEnd.set(Calendar.MILLISECOND, 0);
            
            sameDay = dateFormat.format(calStart.getTime()).equals(dateFormat.format(calEnd.getTime()));
        }
        return sameDay;
    }
    
//    /**
//     * 
//     * @param portalControllerContext
//     * @param document
//     * @param calendarColor
//     * @return
//     * @throws PortletException
//     */
//    private ToutaticeCalendarColor getColor(PortalControllerContext portalControllerContext, Document document, ToutaticeCalendarColor calendarColor) throws PortletException {
//        // Color identifier
//        String colorId;
//        if (document == null) {
//            colorId = null;
//        } else {
//            colorId = document.getString(COLOR_PROPERTY);
//        }
//
//        if ((colorId == null) && (calendarColor != null)) {
//            colorId = ((CalendarColor) calendarColor).getId();
//        }
//
//        return ToutaticeCalendarColor.fromId(colorId);
//    }

}
