package fr.toutatice.services.calendar.view.portlet.controller;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_22;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_29;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_35;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_56;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_ACADEMIE;
import static org.osivia.services.calendar.view.portlet.service.CalendarViewService.DATE_PARAMETER;
import static org.osivia.services.calendar.view.portlet.service.CalendarViewService.PERIOD_TYPE_PARAMETER;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.automation.client.model.Document;
import org.osivia.portal.api.batch.IBatchService;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.internationalization.Bundle;
import org.osivia.portal.api.notifications.NotificationsType;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;
import org.osivia.services.calendar.view.portlet.controller.ViewCalendarController;
import org.osivia.services.calendar.view.portlet.model.CalendarEditionMode;
import org.osivia.services.calendar.view.portlet.model.CalendarViewForm;
import org.osivia.services.calendar.view.portlet.model.calendar.CalendarData;
import org.osivia.services.calendar.view.portlet.service.CalendarViewService;
import org.osivia.services.calendar.view.portlet.utils.PeriodTypes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.portlet.bind.annotation.ActionMapping;
import org.springframework.web.portlet.bind.annotation.RenderMapping;
import org.springframework.web.portlet.bind.annotation.ResourceMapping;

import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoException;
import fr.toutatice.portail.cms.nuxeo.api.cms.NuxeoDocumentContext;
import fr.toutatice.services.calendar.common.model.InteractikCalendarColor;
import fr.toutatice.services.calendar.view.portlet.model.InteractikFilters;
import fr.toutatice.services.calendar.view.portlet.model.calendar.InteractikCalendarData;
import fr.toutatice.services.calendar.view.portlet.service.InteractikCalendarViewService;

/**
 * View calendar controller.
 *
 * @author Cédric Krommenhoek
 * @author Julien Barberet
 */
@Controller
@RequestMapping(value = "VIEW")
@Primary
public class InteractikViewCalendarController extends ViewCalendarController {

	/** Parameter for the scroll position in view day or view week */
	private static final String SCROLL_DAY_WEEK_PARAMETER = "scrollViewDayWeek";
	/** Parameter for the scroll position in view month */
	private static final String SCROLL_MONTH_PARAMETER = "scrollViewMonth";

	/** Color of the primary calendar */
	private static final String PRIMARY_CALENDAR_COLOR = "cal:color";

	/** Nuxeo document request attribute name. */
	private static final String DOCUMENT_REQUEST_ATTRIBUTE = "osivia.calendar.document";
	/** CMS path window property name. */
	private static final String CMS_PATH_WINDOW_PROPERTY = "osivia.calendar.cmsPath";
	/** Interactik Filter request attribute. */
	protected static final String INTERACTIK_FILTERS = "InteractikFilters";

	/** Calendar service. */
	@Autowired
	protected CalendarViewService calendarService;

	/** Portlet service. */
	@Autowired
	private InteractikCalendarViewService service;

	/** Batch service. */
	@Autowired
	private IBatchService batchService;

	/**
	 * Constructor.
	 */
	public InteractikViewCalendarController() {
		super();
	}

	/**
	 * View calendar render mapping
	 *
	 * @param request      render request
	 * @param response     render response
	 * @param selectedDate request parameter, may be null
	 * @return view path
	 * @throws PortletException
	 */
	@RenderMapping
	public String view(RenderRequest request, RenderResponse response,
			@RequestParam(value = DATE_PARAMETER, required = false) String startDate,
			@RequestParam(value = PERIOD_TYPE_PARAMETER, required = false) String periodType,
			@RequestParam(value = SCROLL_DAY_WEEK_PARAMETER, required = false) String scrollViewDayWeek,
			@RequestParam(value = SCROLL_MONTH_PARAMETER, required = false) String scrollViewMonth,
			@ModelAttribute CalendarData calendarData) throws PortletException {

		InteractikCalendarData interactikCalendarData = (InteractikCalendarData) calendarData;

		PortalControllerContext portalControllerContext = new PortalControllerContext(this.portletContext, request,
				response);
		// Nuxeo controller
		NuxeoController nuxeoController = new NuxeoController(portalControllerContext.getRequest(),
				portalControllerContext.getResponse(), portalControllerContext.getPortletCtx());

		this.getDocument(nuxeoController, portalControllerContext);

		String url = super.view(request, response, startDate, periodType, scrollViewDayWeek, scrollViewMonth,
				interactikCalendarData);

		// Set agenda color for quick creation event
		calendarData.setAgendaBackgroundColor(InteractikCalendarColor.ACADEMIE_BRETAGNE.getBackgroundClass());

		// Rewrite JSP path
		if (StringUtils.equals(url, PeriodTypes.DAY.getViewPath())) {
			url += "-interactik";
		}

		return url;
	}

	/**
	 * Ajax call to load scheduler events
	 *
	 * @param request  resource request
	 * @param response resource response
	 * @param path     parent path, may be null for root node
	 * @throws PortletException
	 * @throws IOException
	 */
	@ResourceMapping(value = "initSchedulerData")
	public void initSchedulerData(ResourceRequest request, ResourceResponse response,
			@ModelAttribute(value = CALENDAR_DATA_ATTRIBUTE) CalendarData calendarData,
			@RequestParam(value = PERIOD_TYPE_PARAMETER, required = false) String periodTypeName,
			@RequestParam(value = DATE_PARAMETER, required = false) Date selectedDate)
			throws PortletException, IOException {
		// Portal controller context
		PortalControllerContext portalControllerContext = new PortalControllerContext(this.portletContext, request,
				response);

		dataLoading(response, portalControllerContext, calendarData);
	}

	/**
	 * Event'save after drag and drop
	 * 
	 * @param request
	 * @param response
	 * @param session
	 * @param startDate
	 * @param endDate
	 * @param docid
	 * @param title
	 * @param scrollViewDayWeek
	 * @param scrollViewMonth
	 * @param period
	 * @param date
	 * @throws PortletException
	 * @throws IOException
	 */
	@ActionMapping("dragndrop")
	@Override
	public void dragndrop(ActionRequest request, ActionResponse response, PortletSession session,
			@RequestParam("start") String startDate, @RequestParam("end") String endDate,
			@RequestParam("doc_id") String docid, @RequestParam("title") String title,
			@RequestParam(value = SCROLL_DAY_WEEK_PARAMETER, required = false) String scrollViewDayWeek,
			@RequestParam(value = SCROLL_MONTH_PARAMETER, required = false) String scrollViewMonth,
			@RequestParam(value = PERIOD_TYPE_PARAMETER) String period,
			@RequestParam(value = DATE_PARAMETER) String date) throws PortletException, IOException {
		// Portal controller context
		PortalControllerContext portalControllerContext = new PortalControllerContext(this.portletContext, request,
				response);

		CalendarViewForm form = new CalendarViewForm();
		DateFormat df1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
		df1.setTimeZone(TimeZone.getTimeZone("GMT"));
		try {
			form.setEndDate(df1.parse(endDate));
			form.setStartDate(df1.parse(startDate));

			form.setTitle(title);
			if ("undefined".equals(docid)) {
				form.setDocId(null);
				form.setMode(CalendarEditionMode.CREATION);
			} else {
				form.setDocId(docid);
				form.setMode(CalendarEditionMode.EDITION);
			}

			// Récupérer l'évènement concerné et modifier sa date de début et de fin
			this.calendarService.save(portalControllerContext, form);
		} catch (ParseException e) {
			// Bundle
			Bundle bundle = this.bundleFactory.getBundle(portalControllerContext.getRequest().getLocale());
			this.notificationsService.addSimpleNotification(portalControllerContext,
					bundle.getString("MESSAGE_EVENT_NOT_SAVE"), NotificationsType.WARNING);
		}

		setResponseParameterInteractik(request.getParameter(COOPERATIVE_22), request.getParameter(COOPERATIVE_29),
				request.getParameter(COOPERATIVE_35), request.getParameter(COOPERATIVE_56),
				request.getParameter(COOPERATIVE_ACADEMIE), response, date, period, scrollViewDayWeek, scrollViewMonth);
	}

	/**
	 * Get Nuxeo document.
	 *
	 * @param nuxeoController Nuxeo controller
	 * @return document
	 * @throws PortletException
	 */
	private Document getDocument(NuxeoController nuxeoController, PortalControllerContext portalControllerContext)
			throws PortletException {
		PortletRequest request = nuxeoController.getRequest();
		Document document = (Document) request.getAttribute(DOCUMENT_REQUEST_ATTRIBUTE);

		if (document == null) {
			try {
				String cmsPath = this.getCMSPath(nuxeoController, portalControllerContext);
				if (StringUtils.isNotEmpty(cmsPath)) {
					NuxeoDocumentContext documentContext = nuxeoController.getDocumentContext(cmsPath);
					document = documentContext.getDocument();
					request.setAttribute(DOCUMENT_REQUEST_ATTRIBUTE, document);
				}
			} catch (NuxeoException e) {
				if (NuxeoException.ERROR_FORBIDDEN != e.getErrorCode()) {
					throw new PortletException(e);
				}
			}
		}

		return document;
	}

	/**
	 * Get CMS path.
	 *
	 * @param nuxeoController Nuxeo controller
	 * @return CMS path
	 * @throws PortletException
	 */
	private String getCMSPath(NuxeoController nuxeoController, PortalControllerContext portalControllerContext)
			throws PortletException {

		PortalWindow window = WindowFactory.getWindow(portalControllerContext.getRequest());
		// Context path
		String cmsPath = window.getProperty(CMS_PATH_WINDOW_PROPERTY);
		if (!StringUtils.isNotBlank(cmsPath)) {
			// Current window
			window = WindowFactory.getWindow(nuxeoController.getRequest());
			cmsPath = window.getPageProperty("osivia.cms.basePath");
		}

		return nuxeoController.getComputedPath(cmsPath);
	}

	private void setResponseParameterInteractik(String coop22, String coop29, String coop35, String coop56,
			String coopAcademie, ActionResponse response, String date, String period, String scrollViewDayWeek,
			String scrollViewMonth) {
		response.setRenderParameter(COOPERATIVE_22, coop22);
		response.setRenderParameter(COOPERATIVE_29, coop29);
		response.setRenderParameter(COOPERATIVE_35, coop35);
		response.setRenderParameter(COOPERATIVE_56, coop56);
		response.setRenderParameter(COOPERATIVE_ACADEMIE, coopAcademie);
		setResponseParameter(response, date, period, scrollViewDayWeek, scrollViewMonth);
	}

	/**
	 * Add event action mapping.
	 * 
	 * @param request  action request
	 * @param response action response
	 * @throws PortletException
	 * @throws IOException
	 */
	@ActionMapping("addEvent")
	public void addEvent(ActionRequest request, ActionResponse response) throws PortletException, IOException {
		// Portal controller context
		PortalControllerContext portalControllerContext = new PortalControllerContext(this.portletContext, request,
				response);

		// Redirection URL
		String redirectionUrl = this.service.getEventCreationRedirectionUrl(portalControllerContext);

		response.sendRedirect(redirectionUrl);
	}

	/**
	 * Search coop action mapping.
	 * 
	 * @param request  action request
	 * @param response action response
	 * @throws PortletException
	 * @throws IOException
	 */
	@ActionMapping("search")
	public void search(ActionRequest request, ActionResponse response,
			@RequestParam(value = PERIOD_TYPE_PARAMETER) String period,
			@RequestParam(value = DATE_PARAMETER) String date,
			@ModelAttribute(INTERACTIK_FILTERS) InteractikFilters filters) throws PortletException, IOException {

		response.setRenderParameter(COOPERATIVE_22, String.valueOf(filters.isCooperative22()));
		response.setRenderParameter(COOPERATIVE_29, String.valueOf(filters.isCooperative29()));
		response.setRenderParameter(COOPERATIVE_35, String.valueOf(filters.isCooperative35()));
		response.setRenderParameter(COOPERATIVE_56, String.valueOf(filters.isCooperative56()));
		response.setRenderParameter(COOPERATIVE_ACADEMIE, String.valueOf(filters.isCooperativeAcademie()));

		response.setRenderParameter(DATE_PARAMETER, date);
		response.setRenderParameter(PERIOD_TYPE_PARAMETER, period);

	}

	/**
	 * Get Filter Interactik.
	 *
	 * @param request  portlet request
	 * @param response portlet response
	 * @return calendar data
	 * @throws PortletException
	 */
	@ModelAttribute(value = INTERACTIK_FILTERS)
	public InteractikFilters getInteractikFilters(PortletRequest request, PortletResponse response)
			throws PortletException {
		// Portal controller context
		PortalControllerContext portalControllerContext = new PortalControllerContext(this.portletContext, request,
				response);

		return this.service.getFilters(portalControllerContext);
	}
}
