package fr.toutatice.services.calendar.view.portlet.repository;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.AGENDA_INTERACTIK;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.CODE_DEPARTEMENT_ACADEMIE;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_22;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_29;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_35;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_56;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.COOPERATIVE_ACADEMIE;
import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DEPARTEMENT_PROPERTY;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.portlet.PortletException;
import javax.portlet.PortletRequest;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.automation.client.model.Document;
import org.nuxeo.ecm.automation.client.model.Documents;
import org.nuxeo.ecm.automation.client.model.PropertyList;
import org.osivia.portal.api.context.PortalControllerContext;
import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;
import org.osivia.services.calendar.edition.portlet.model.CalendarSynchronizationSource;
import org.osivia.services.calendar.view.portlet.model.CalendarEditionMode;
import org.osivia.services.calendar.view.portlet.model.CalendarOptions;
import org.osivia.services.calendar.view.portlet.model.CalendarViewForm;
import org.osivia.services.calendar.view.portlet.model.events.Event;
import org.osivia.services.calendar.view.portlet.model.events.EventKey;
import org.osivia.services.calendar.view.portlet.model.events.EventToSync;
import org.osivia.services.calendar.view.portlet.repository.CalendarViewRepositoryImpl;
import org.osivia.services.calendar.view.portlet.repository.command.EventGetCommand;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Repository;

import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoController;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoQueryFilterContext;
import fr.toutatice.portail.cms.nuxeo.api.cms.NuxeoDocumentContext;
import fr.toutatice.services.calendar.common.InteractikCalendarConstant;
import fr.toutatice.services.calendar.common.model.InteractikCalendarColor;
import fr.toutatice.services.calendar.view.portlet.model.Cooperative;
import fr.toutatice.services.calendar.view.portlet.model.InteractikCalendarOptions;
import fr.toutatice.services.calendar.view.portlet.model.events.InteractikEvent;
import fr.toutatice.services.calendar.view.portlet.repository.command.InteractikEventEditionCommand;
import fr.toutatice.services.calendar.view.portlet.repository.command.InteractikEventListCommand;
import fr.toutatice.services.calendar.view.portlet.repository.command.InteractikSynchronizationCommand;

/**
 * Calendar repository implementation.
 *
 * @author Julien Barberet
 * @author Cédric Krommenhoek
 * @see CalendarViewRepositoryImpl
 */
@Repository
@Primary
public class InteractikCalendarViewRepositoryImpl extends CalendarViewRepositoryImpl {

    /** Cooperative window property name. */
    private static final String COOPERATIVE_WINDOW_PROPERTY = "interactik.calendar.cooperative";


    /** Application context. */
    @Autowired
    private ApplicationContext applicationContext;


    /**
     * Constructor.
     */
    public InteractikCalendarViewRepositoryImpl() {
        super();

    }


    /**
     * {@inheritDoc}
     */
    @Override
    public CalendarOptions getConfiguration(PortalControllerContext portalControllerContext) throws PortletException {
        // Options
        CalendarOptions options = super.getConfiguration(portalControllerContext);

        if (options instanceof InteractikCalendarOptions) {
            InteractikCalendarOptions customizedOptions = (InteractikCalendarOptions) options;

            // Current window
            PortalWindow window = WindowFactory.getWindow(portalControllerContext.getRequest());

            // Coopérative sélectionnée
            Cooperative selectedCooperative = Cooperative.fromName(window.getProperty(COOPERATIVE_WINDOW_PROPERTY));
            customizedOptions.setSelectedCooperative(selectedCooperative);
        }

        return options;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void saveConfiguration(PortalControllerContext portalControllerContext, CalendarOptions options) throws PortletException {
        super.saveConfiguration(portalControllerContext, options);

        if (options instanceof InteractikCalendarOptions) {
            InteractikCalendarOptions customizedOptions = (InteractikCalendarOptions) options;

            // Current window
            PortalWindow window = WindowFactory.getWindow(portalControllerContext.getRequest());

            // Coopérative sélectionnée
            Cooperative selectedCooperative = customizedOptions.getSelectedCooperative();
            if ((selectedCooperative == null) || !selectedCooperative.isDisplayed()) {
                window.setProperty(COOPERATIVE_WINDOW_PROPERTY, null);
            } else {
                window.setProperty(COOPERATIVE_WINDOW_PROPERTY, selectedCooperative.name());
            }

        }
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Event getEvent(PortalControllerContext portalControllerContext, String docid) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext.getRequest(), portalControllerContext.getResponse(),
                portalControllerContext.getPortletCtx());
        // CMS path
        String cmsPath = this.getCMSPath(nuxeoController);
        INuxeoCommand nuxeoCommand = new EventGetCommand(cmsPath, docid);
        Document document = (Document) nuxeoController.executeNuxeoCommand(nuxeoCommand);

        return this.fillEvent(document, nuxeoController);
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void synchronize(PortalControllerContext portalControllerContext, Map<EventKey, EventToSync> map) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);

        // CMS path
        String cmsPath = this.getCMSPath(nuxeoController);
        String parentPath = nuxeoController.getContentPath();

        INuxeoCommand nuxeoCommand = new InteractikSynchronizationCommand(NuxeoQueryFilterContext.CONTEXT_LIVE_N_PUBLISHED, cmsPath, parentPath, (map));
        nuxeoController.executeNuxeoCommand(nuxeoCommand);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void save(PortalControllerContext portalControllerContext, CalendarViewForm form) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext);

        // CMS path
        String cmsPath = this.getCMSPath(nuxeoController);
        form.setParentPath(cmsPath);

        // Nuxeo command
        INuxeoCommand command = this.applicationContext.getBean(InteractikEventEditionCommand.class, form);
        Document document = (Document) nuxeoController.executeNuxeoCommand(command);

        if (CalendarEditionMode.EDITION.equals(form.getMode())) {
            // Refresh document
            NuxeoDocumentContext documentContext = nuxeoController.getDocumentContext(document.getPath());
            documentContext.reload();
            document = documentContext.getDocument();
        }

        form.setDocument(document);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Event> getEvents(PortalControllerContext portalControllerContext, Date start, Date end) throws PortletException {
        // Nuxeo controller
        NuxeoController nuxeoController = new NuxeoController(portalControllerContext.getRequest(), portalControllerContext.getResponse(),
                portalControllerContext.getPortletCtx());
        PortletRequest request = portalControllerContext.getRequest();

        boolean agendaInteractik = false;
        if (request.getAttribute(AGENDA_INTERACTIK) != null) {
            agendaInteractik = BooleanUtils.toBoolean((Boolean) request.getAttribute(AGENDA_INTERACTIK));
            request.removeAttribute(AGENDA_INTERACTIK);
        }

        boolean cooperative22 = false;
        boolean cooperative29 = false;
        boolean cooperative35 = false;
        boolean cooperative56 = false;
        boolean cooperativeAcademie = false;

        if (StringUtils.isEmpty(request.getParameter("format"))) {
            // Options
            CalendarOptions options = this.getConfiguration(portalControllerContext);
            if (options instanceof InteractikCalendarOptions) {
                InteractikCalendarOptions interactikOptions = (InteractikCalendarOptions) options;

                // Coopératives
                Cooperative selectedCooperative = interactikOptions.getSelectedCooperative();
                if ((selectedCooperative == null) || !selectedCooperative.isDisplayed()) {
                    if (request.getAttribute(COOPERATIVE_22) != null) {
                        cooperative22 = BooleanUtils.toBoolean(request.getParameter(COOPERATIVE_22));
                        request.removeAttribute(COOPERATIVE_22);
                    }
                    if (request.getAttribute(COOPERATIVE_29) != null) {
                        cooperative29 = BooleanUtils.toBoolean(request.getParameter(COOPERATIVE_29));
                        request.removeAttribute(COOPERATIVE_29);
                    }
                    if (request.getAttribute(COOPERATIVE_35) != null) {
                        cooperative35 = BooleanUtils.toBoolean(request.getParameter(COOPERATIVE_35));
                        request.removeAttribute(COOPERATIVE_35);
                    }
                    if (request.getAttribute(COOPERATIVE_56) != null) {
                        cooperative56 = BooleanUtils.toBoolean(request.getParameter(COOPERATIVE_56));
                        request.removeAttribute(COOPERATIVE_56);
                    }
                    if (request.getAttribute(COOPERATIVE_ACADEMIE) != null) {
                        cooperativeAcademie = BooleanUtils.toBoolean(request.getParameter(COOPERATIVE_ACADEMIE));
                        request.removeAttribute(COOPERATIVE_ACADEMIE);
                    }
                } else if (Cooperative.COTES_ARMOR.equals(selectedCooperative)) {
                    cooperative22 = true;
                } else if (Cooperative.FINISTERE.equals(selectedCooperative)) {
                    cooperative29 = true;
                } else if (Cooperative.ILLE_ET_VILAINE.equals(selectedCooperative)) {
                    cooperative35 = true;
                } else if (Cooperative.MORBIHAN.equals(selectedCooperative)) {
                    cooperative56 = true;
                } else if (Cooperative.ACADEMIE.equals(selectedCooperative)) {
                    cooperativeAcademie = true;
                }
            }
        } else {
            agendaInteractik = true;
            String filtersParameter = request.getParameter("filters");
            if (StringUtils.isEmpty(filtersParameter)) {
                cooperative22 = true;
                cooperative29 = true;
                cooperative35 = true;
                cooperative56 = true;
                cooperativeAcademie = true;
            } else {
                String[] filters = StringUtils.split(filtersParameter, "&");
                for (String filter : filters) {
                    String[] entry = StringUtils.split(filter, "=");
                    if (ArrayUtils.isNotEmpty(entry) && (entry.length == 2)) {
                        String name = entry[0];
                        String value = entry[1];

                        if (StringUtils.equals("agenda", name)) {
                            Cooperative cooperative = Cooperative.fromName(value);

                            if (Cooperative.COTES_ARMOR.equals(cooperative)) {
                                cooperative22 = true;
                            } else if (Cooperative.FINISTERE.equals(cooperative)) {
                                cooperative29 = true;
                            } else if (Cooperative.ILLE_ET_VILAINE.equals(cooperative)) {
                                cooperative35 = true;
                            } else if (Cooperative.MORBIHAN.equals(cooperative)) {
                                cooperative56 = true;
                            } else if (Cooperative.ACADEMIE.equals(cooperative)) {
                                cooperativeAcademie = true;
                            }
                        }
                    }
                }
            }
        }


        // CMS path
        String cmsPath = this.getCMSPath(nuxeoController);

        List<Event> events;
        if (StringUtils.isEmpty(cmsPath)) {
            events = null;
        } else {
            List<CalendarSynchronizationSource> listSource = this.getSynchronizationSources(portalControllerContext);

            // Nuxeo command
            INuxeoCommand nuxeoCommand = new InteractikEventListCommand(NuxeoQueryFilterContext.CONTEXT_LIVE_N_PUBLISHED, cmsPath, start, end, listSource,
                    cooperative22, cooperative29, cooperative35, cooperative56, cooperativeAcademie, agendaInteractik);
            Documents documents = (Documents) nuxeoController.executeNuxeoCommand(nuxeoCommand);

            // Events
            events = new ArrayList<Event>(documents.size());

            for (Document document : documents) {
                if ((document.getDate(START_DATE_PROPERTY) != null) && (document.getDate(END_DATE_PROPERTY) != null)) {
                    // Event
                    Event event = this.fillEvent(document, nuxeoController);

                    events.add(event);
                }
            }
        }

        return events;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    protected Event fillEvent(Document document, NuxeoController nuxeoController) {
        Event event = super.fillEvent(document, nuxeoController);
        
        // Background color
        String bckgcolor = document.getString(BCKG_COLOR);
        String dept = document.getString(DEPARTEMENT_PROPERTY);
        if (dept != null) {
            if (CODE_DEPARTEMENT_ACADEMIE.equals(dept)) {
                bckgcolor = InteractikCalendarColor.ACADEMIE_BRETAGNE.toString();
            } else if ("22".equals(dept)) {
                bckgcolor = InteractikCalendarColor.COTE_D_ARMOR.toString();
            } else if ("29".equals(dept)) {
                bckgcolor = InteractikCalendarColor.FINISTERE.toString();
            } else if ("35".equals(dept)) {
                bckgcolor = InteractikCalendarColor.ILE_ET_VILAINE.toString();
            } else if ("56".equals(dept)) {
                bckgcolor = InteractikCalendarColor.MORBIHAN.toString();
            }
        }
        // Lorque les événements sont passés on les grise
        Date date = new Date();
        SimpleDateFormat fmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        String dateJour = fmt.format(date);
        if(dateJour.compareTo(document.getString(END_DATE_PROPERTY).substring(0, 19)) > 0) {
        	bckgcolor = InteractikCalendarColor.GREY.toString();
        }
        event.setBckgColor(bckgcolor);
        
        if (event instanceof InteractikEvent) {
            InteractikEvent interactikEvent = (InteractikEvent) event;

            // Département
            String departement = document.getString(InteractikCalendarConstant.DEPARTEMENT_PROPERTY);
            interactikEvent.setDepartement(departement);
            
            // Ville
            String ville = document.getString(InteractikCalendarConstant.VILLE_PROPERTY);
            interactikEvent.setVille(ville);
            
            // URL inscription
            String urlInscription = document.getString(InteractikCalendarConstant.URL_INSCRIPTION_PROPERTY);
            interactikEvent.setUrlInscription(urlInscription);

            // Date de début d'inscription
            Date debutInscription = document.getDate(InteractikCalendarConstant.DATE_DEBUT_INSCRIPTION_PROPERTY);
            interactikEvent.setDebutInscription(debutInscription);

            // Date de fin d'inscription
            Date finInscription = document.getDate(InteractikCalendarConstant.DATE_FIN_INSCRIPTION_PROPERTY);
            interactikEvent.setFinInscription(finInscription);
            
            // Organisateurs
            List<String> organisateurs = this.getEventProperties(document, InteractikCalendarConstant.ORGANISATEURS_PROPERTY);
            interactikEvent.setOrganisateurs(organisateurs);

            // Pôles disciplinaires
            List<String> polesDisciplinaires = this.getEventProperties(document, InteractikCalendarConstant.POLES_DISCIPLINAIRES_PROPERTY);
            interactikEvent.setPolesDisciplinaires(polesDisciplinaires);

            // Thèmes
            List<String> themes = this.getEventProperties(document, InteractikCalendarConstant.THEMES_PROPERTY);
            interactikEvent.setThemes(themes);

            // Niveaux
            List<String> niveaux = this.getEventProperties(document, InteractikCalendarConstant.NIVEAUX_PROPERTY);
            interactikEvent.setNiveaux(niveaux);

            // Contenus
            List<String> contenus = this.getEventProperties(document, InteractikCalendarConstant.CONTENUS_PROPERTY);
            interactikEvent.setContenus(contenus);

            // Repères DNE
            List<String> reperesDne = this.getEventProperties(document, InteractikCalendarConstant.REPERES_DNE_PROPERTY);
            interactikEvent.setReperesDne(reperesDne);
            
        }

        return event;
    }


    /**
     * Get event properties.
     * 
     * @param document Nuxeo document
     * @param propertyName document property name
     * @return properties
     */
    private List<String> getEventProperties(Document document, String propertyName) {
        // Document property list
        PropertyList list = document.getProperties().getList(propertyName);

        // Properties
        List<String> properties;
        if ((list == null) || list.isEmpty()) {
            properties = null;
        } else {
            properties = new ArrayList<>(list.size());
            for (int i = 0; i < list.size(); i++) {
                String value = list.getString(i);
                properties.add(value);
            }
        }

        return properties;
    }

}
