package fr.toutatice.services.calendar.view.portlet.repository.command;

import static fr.toutatice.services.calendar.common.InteractikCalendarConstant.DEPARTEMENT_PROPERTY;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.nuxeo.ecm.automation.client.Constants;
import org.nuxeo.ecm.automation.client.OperationRequest;
import org.nuxeo.ecm.automation.client.Session;
import org.osivia.services.calendar.edition.portlet.model.CalendarSynchronizationSource;

import fr.toutatice.portail.cms.nuxeo.api.INuxeoCommand;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoCompatibility;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoQueryFilter;
import fr.toutatice.portail.cms.nuxeo.api.NuxeoQueryFilterContext;

/**
 * List Nuxeo events command.
 *
 * @author Cédric Krommenhoek
 * @author Julien Barberet
 * @see INuxeoCommand
 */
public class InteractikEventListCommand implements INuxeoCommand {

    /** Nuxeo query filter context. */
    private NuxeoQueryFilterContext queryContext;
    /** Context path. */
    private final String contextPath;
    /** Start date. */
    private final Date startDate;
    /** End date. */
    private final Date endDate;
    
    private final String sourcesId;
    
    private final boolean cooperative22;
    
    private final boolean cooperative29;
    
    private final boolean cooperative35;
    
    private final boolean cooperative56;
    
    private final boolean cooperativeAcademie;
    
    private final boolean agendaInteractik;


    /**
     * Constructor.
     *
     * @param queryContext Nuxeo query filter context
     * @param contextPath context path
     * @param startDate start date
     * @param endDate end date
     */
    public InteractikEventListCommand(NuxeoQueryFilterContext queryContext, String contextPath, Date startDate, Date endDate, List<CalendarSynchronizationSource> listSource,
    		boolean cooperative22, boolean cooperative29, boolean cooperative35, boolean cooperative56, boolean cooperativeAcademie, boolean agendaInteractik) {
        super();
        this.queryContext = queryContext;
        this.contextPath = contextPath;
        this.startDate = startDate;
        this.endDate = endDate;
        this.sourcesId = getStringValues(listSource);
        this.cooperative22 = cooperative22;
        this.cooperative29 = cooperative29;
        this.cooperative35 = cooperative35;
        this.cooperative56 = cooperative56;
        this.cooperativeAcademie = cooperativeAcademie;
        this.agendaInteractik = agendaInteractik;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Object execute(Session nuxeoSession) throws Exception {
        String start = DateFormatUtils.ISO_DATETIME_TIME_ZONE_FORMAT.format(this.startDate);

        String end;
        if (this.endDate == null) {
            end = null;
        } else {
            end = DateFormatUtils.ISO_DATETIME_TIME_ZONE_FORMAT.format(this.endDate);
        }

        // Clause
        StringBuilder clause = new StringBuilder();
        clause.append("ecm:mixinType = 'Schedulable' ");
        clause.append("AND ecm:path STARTSWITH '").append(this.contextPath).append("' ");
        if (StringUtils.isNotEmpty(end)) {
            clause.append("AND (vevent:dtstart < TIMESTAMP '").append(end).append("') ");
        }
        clause.append("AND (vevent:dtend > TIMESTAMP '").append(start).append("') ");
        if (agendaInteractik)
        {
	        boolean comma = false;
	        StringBuilder clauseDepartement = new StringBuilder();
	        if (cooperative22)
	        {
	        	clauseDepartement.append("'22'");
	        	comma = true;
	        }
	        if (cooperative29)
	        {
	        	clauseDepartement.append(comma? ",'29'" : "'29'");
	        	comma = true;
	        }
	        if (cooperative35)
	        {
	        	clauseDepartement.append(comma? ",'35'" : "'35'");
	        	comma = true;
	        }
	        if (cooperative56)
	        {
	        	clauseDepartement.append(comma? ",'56'" : "'56'");
	        	comma = true;
	        }
	        if (cooperativeAcademie)
	        {
	        	clauseDepartement.append(comma? ",'aca'" : "'aca'");
	        }
	        if (clauseDepartement.length()>0)
	        {
	        	clause.append("AND ").append(DEPARTEMENT_PROPERTY).append(" in (").append(clauseDepartement).append(") ");
	        } else
	        {
	        	//If no department selected, no event returned
	        	clause.append("AND ").append(DEPARTEMENT_PROPERTY).append(" = '1' AND ").append(DEPARTEMENT_PROPERTY).append(" = '2' ");
	        }
        }
        clause.append("AND (sync:idParentSource is null or sync:idParentSource = '' ");
        if (this.sourcesId != null && !this.sourcesId.isEmpty()) clause.append(" OR sync:idParentSource IN (").append(this.sourcesId).append(")");
        clause.append(") ");
        clause.append("ORDER BY vevent:dtstart");

        // Filter on published documents
        String filteredRequest = NuxeoQueryFilter.addPublicationFilter(this.queryContext, clause.toString());

        // Request
        OperationRequest request;
        if (NuxeoCompatibility.canUseES()) {
            request = nuxeoSession.newRequest("Document.QueryES");
            request.set(Constants.HEADER_NX_SCHEMAS, "*");
        } else {
            request = nuxeoSession.newRequest("Document.Query");
            request.setHeader(Constants.HEADER_NX_SCHEMAS, "*");
        }
        request.set("query", "SELECT * FROM Document WHERE " + filteredRequest);

        return request.execute();
    }

    /**
     * Get string values.
     * 
     * @param values values
     * @return string
     */
    private String getStringValues(List<CalendarSynchronizationSource> values) {
        String result;

        if (values == null) {
            result = null;
        } else {
            StringBuilder builder = new StringBuilder();

            boolean first = true;
            for (CalendarSynchronizationSource value : values) {
                if (first) {
                    first = false;
                } else {
                    builder.append(", ");
                }

                builder.append("'");
                builder.append(value.getId());
                builder.append("'");
            }

            result = builder.toString();
        }

        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getId() {
        return "Calendar/" + this.contextPath;
    }

}
