/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.common.file;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.common.file.FileCache;

public class LRUFileCache
implements FileCache {
    private static final Log log = LogFactory.getLog(LRUFileCache.class);
    public static final Pattern SIMPLE_ASCII = Pattern.compile("[-_a-zA-Z0-9]+");
    private static final String TMP_PREFIX = "nxbin_";
    private static final String TMP_SUFFIX = ".tmp";
    public static long MIN_AGE_MILLIS = 3600000L;
    public static long CLEAR_OLD_ENTRIES_INTERVAL_MILLIS = 5000L;
    protected final Path dir;
    protected final long maxSize;
    protected Lock clearOldEntriesLock = new ReentrantLock();
    protected long clearOldEntriesLast;

    public LRUFileCache(File dir, long maxSize) {
        this.dir = dir.toPath();
        this.maxSize = maxSize;
    }

    @Override
    public long getSize() {
        long size = 0L;
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(this.dir, RegularFileFilter.INSTANCE);){
            for (Path path : ds) {
                size += Files.size(path);
            }
        }
        catch (IOException e) {
            log.error((Object)e, (Throwable)e);
        }
        return size;
    }

    @Override
    public int getNumberOfItems() {
        int count = 0;
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(this.dir, RegularFileFilter.INSTANCE);){
            for (Path path : ds) {
                ++count;
            }
        }
        catch (IOException e) {
            log.error((Object)e, (Throwable)e);
        }
        return count;
    }

    @Override
    public void clear() {
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(this.dir, RegularFileFilter.INSTANCE);){
            for (Path path : ds) {
                try {
                    Files.delete(path);
                }
                catch (IOException e) {
                    log.error((Object)e, (Throwable)e);
                }
            }
        }
        catch (IOException e) {
            log.error((Object)e, (Throwable)e);
        }
    }

    protected void clearOldEntries() {
        if (this.clearOldEntriesLock.tryLock()) {
            try {
                if (System.currentTimeMillis() > this.clearOldEntriesLast + CLEAR_OLD_ENTRIES_INTERVAL_MILLIS) {
                    this.doClearOldEntries();
                    this.clearOldEntriesLast = System.currentTimeMillis();
                    return;
                }
            }
            finally {
                this.clearOldEntriesLock.unlock();
            }
        }
    }

    protected void doClearOldEntries() {
        ArrayList<PathInfo> files = new ArrayList<PathInfo>();
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(this.dir, RegularFileFilter.INSTANCE);){
            for (Path path : ds) {
                try {
                    files.add(new PathInfo(path));
                }
                catch (IOException e) {
                    log.error((Object)e, (Throwable)e);
                }
            }
        }
        catch (IOException e) {
            log.error((Object)e, (Throwable)e);
        }
        Collections.sort(files);
        long size = 0L;
        long threshold = System.currentTimeMillis() - MIN_AGE_MILLIS;
        for (PathInfo pi : files) {
            if (pi.time >= threshold || (size += pi.size) <= this.maxSize) continue;
            try {
                Files.delete(pi.path);
                size -= pi.size;
            }
            catch (IOException e) {
                log.error((Object)e, (Throwable)e);
            }
        }
    }

    @Override
    public File getTempFile() throws IOException {
        return Files.createTempFile(this.dir, TMP_PREFIX, TMP_SUFFIX, new FileAttribute[0]).toFile();
    }

    protected void checkKey(String key) throws IllegalArgumentException {
        if (!SIMPLE_ASCII.matcher(key).matches() || ".".equals(key) || "..".equals(key)) {
            throw new IllegalArgumentException("Invalid key: " + key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public File putFile(String key, InputStream in) throws IOException {
        File tmp;
        try {
            this.checkKey(key);
            Path path = this.dir.resolve(key);
            if (Files.exists(path, new LinkOption[0])) {
                this.recordAccess(path);
                File file = path.toFile();
                return file;
            }
            tmp = this.getTempFile();
            try (FileOutputStream out = new FileOutputStream(tmp);){
                IOUtils.copy((InputStream)in, (OutputStream)out);
            }
        }
        finally {
            in.close();
        }
        return this.putFile(key, tmp);
    }

    @Override
    public File putFile(String key, File file) throws IllegalArgumentException, IOException {
        Path source = file.toPath();
        this.checkKey(key);
        Path path = this.dir.resolve(key);
        try {
            Files.move(source, path, new CopyOption[0]);
            this.recordAccess(path);
            this.clearOldEntries();
        }
        catch (FileAlreadyExistsException faee) {
            this.recordAccess(path);
            try {
                Files.delete(source);
            }
            catch (IOException e) {
                log.error((Object)e, (Throwable)e);
            }
        }
        return path.toFile();
    }

    @Override
    public File getFile(String key) {
        this.checkKey(key);
        Path path = this.dir.resolve(key);
        if (!Files.exists(path, new LinkOption[0])) {
            return null;
        }
        this.recordAccess(path);
        return path.toFile();
    }

    protected void recordAccess(Path path) {
        try {
            Files.setLastModifiedTime(path, FileTime.fromMillis(System.currentTimeMillis()));
        }
        catch (IOException e) {
            log.error((Object)e, (Throwable)e);
        }
    }

    protected static class RegularFileFilter
    implements DirectoryStream.Filter<Path> {
        protected static final RegularFileFilter INSTANCE = new RegularFileFilter();

        protected RegularFileFilter() {
        }

        @Override
        public boolean accept(Path path) {
            if (!Files.isRegularFile(path, new LinkOption[0])) {
                return false;
            }
            String filename = path.getFileName().toString();
            return !filename.startsWith(LRUFileCache.TMP_PREFIX) || !filename.endsWith(LRUFileCache.TMP_SUFFIX);
        }
    }

    protected static class PathInfo
    implements Comparable<PathInfo> {
        protected final Path path;
        protected final long time;
        protected final long size;

        public PathInfo(Path path) throws IOException {
            this.path = path;
            this.time = Files.getLastModifiedTime(path, new LinkOption[0]).toMillis();
            this.size = Files.size(path);
        }

        @Override
        public int compareTo(PathInfo other) {
            return Long.compare(other.time, this.time);
        }
    }
}

