package org.osivia.portal.services.inscription;

import java.util.Date;
import java.util.Properties;
import java.util.logging.Logger;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.GenericPortlet;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.validation.Valid;

import org.osivia.portal.api.windows.PortalWindow;
import org.osivia.portal.api.windows.WindowFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.portlet.bind.annotation.ActionMapping;
import org.springframework.web.portlet.bind.annotation.RenderMapping;
import org.springframework.web.portlet.context.PortletConfigAware;
import org.springframework.web.portlet.context.PortletContextAware;

import com.sun.mail.smtp.SMTPTransport;

/**
 * Contrôleur formulaire de contact
 * 
 */
@Controller
@RequestMapping("VIEW")
@SessionAttributes({"infosForm"})
public class DemandeInfosControleur extends GenericPortlet implements PortletContextAware, PortletConfigAware {

    public final static Logger log = Logger.getLogger(DemandeInfosControleur.class.getSimpleName());

    private PortletContext portletContext;
    private PortletConfig portletConfig;

    public void setPortletConfig(PortletConfig portletConfig) {
        this.portletConfig = portletConfig;

    }

    public void setPortletContext(PortletContext portletContext) {
        this.portletContext = portletContext;

    }

    @Autowired
    private InfosFormValidator infosFormValidator;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        binder.setValidator(infosFormValidator);
    }

    /**
     * Soumission du mail
     * 
     * @param infosForm
     * @param result
     * @param session
     * @param modelMap
     */
    @ActionMapping
    public void envoiMail(@Valid InfosForm infosForm, BindingResult result, ActionRequest request, ActionResponse response, PortletSession session,
            ModelMap modelMap) {

        if (!result.hasErrors()) {
            // Récupération des propriétés systemes (configurés dans le portal.properties).
            Properties props = System.getProperties();

            Session mailSession = Session.getInstance(props, null);

            // Nouveau message
            final MimeMessage msg = new MimeMessage(mailSession);

            // -- Set the FROM and TO fields --
            try {

                msg.setFrom(new InternetAddress(infosForm.getMail()));

                PortalWindow window = WindowFactory.getWindow(request);

                String mailDestinataire = window.getProperty(AdminController.DESTINATAIRE);


                msg.setRecipients(Message.RecipientType.TO, InternetAddress.parse(mailDestinataire, false));

                msg.setSubject("Demande de création de compte", "UTF-8");

                Multipart mp = new MimeMultipart();
                MimeBodyPart htmlPart = new MimeBodyPart();
                htmlPart.setContent(DemandeInfosFormFormatter.format(infosForm), "text/html; charset=UTF-8");
                mp.addBodyPart(htmlPart);

                msg.setContent(mp);

                msg.setSentDate(new Date());

                SMTPTransport t = (SMTPTransport) mailSession.getTransport("smtp");

                t.connect("localhost", "contact", "");

                t.sendMessage(msg, msg.getAllRecipients());
                t.close();
            } catch (AddressException e) {
                log.info(e.getMessage());
            } catch (MessagingException e) {
                log.info(e.getMessage());
            }


            response.setRenderParameter("action", "mailok");
        }

    }

    /**
     * Affichage du formulaire de contact
     * 
     * @param request
     * @param model
     * @return
     */
    @RenderMapping
    public String view(RenderRequest request, ModelMap model) {

        InfosForm infosForm = (InfosForm) model.get("infosForm");

        PortalWindow window = WindowFactory.getWindow(request);

        String mailDestinataire = window.getProperty(AdminController.DESTINATAIRE);

        if (mailDestinataire == null || mailDestinataire.length() == 0) {
            return "no-conf";
        } else {
            return "list";
        }
    }

    /**
     * Affichage écran de confirmation
     * 
     * @param request
     * @param model
     * @return
     */
    @RenderMapping(params = {"action=mailok"})
    public String mailok(RenderRequest request, ModelMap model) {
        return "mailok";
    }

    /**
     * Initialisation d'un modèle vide
     * 
     * @return form
     */
    @ModelAttribute
    private InfosForm loadModel() {

        return new InfosForm();
    }

}
